"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEuiBreakpoint = exports.euiBreakpoint = void 0;

var _sorting = require("../../services/breakpoint/_sorting");

var _hooks = require("../../services/theme/hooks");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Generates a CSS media query rule string based on the input breakpoint ranges.
 * Examples with default theme breakpoints:
 *
 * euiBreakpoint(['s']) becomes `@media only screen and (min-width: 575px) and (max-width: 767px)`
 * euiBreakpoint(['s', 'l']) becomes `@media only screen and (min-width: 575px) and (max-width: 1199px)`
 *
 * Use the smallest and largest sizes to generate media queries with only min/max-width.
 * Examples with default theme breakpoints:
 *
 * euiBreakpoint(['xs', 'm']) becomes `@media only screen and (max-width: 991px)`
 * euiBreakpoint(['l', 'xl']) becomes `@media only screen and (min-width: 992px)`
 */
var euiBreakpoint = function euiBreakpoint(_ref, sizes) {
  var euiTheme = _ref.euiTheme;
  // Ensure we inherit any theme breakpoint overrides & sort by small to large
  var orderedBreakpoints = Object.keys((0, _sorting.sortMapBySmallToLargeValues)(euiTheme.breakpoint)); // Ensure the sizes array is in the correct ascending size order

  var orderedSizes = sizes.sort(function (a, b) {
    return orderedBreakpoints.indexOf(a) - orderedBreakpoints.indexOf(b);
  });
  var firstBreakpoint = orderedSizes[0];
  var minBreakpointSize = euiTheme.breakpoint[firstBreakpoint];
  var lastBreakpoint = orderedSizes[sizes.length - 1];
  var maxBreakpointSize; // To get the correct screen range, we set the max-width to the next breakpoint
  // size in the sizes array (unless the size is already the largest breakpoint)

  if (lastBreakpoint !== orderedBreakpoints[orderedBreakpoints.length - 1]) {
    var nextBreakpoint = orderedBreakpoints.indexOf(lastBreakpoint) + 1;
    maxBreakpointSize = euiTheme.breakpoint[orderedBreakpoints[nextBreakpoint]];
  }

  return ['@media only screen', minBreakpointSize ? "(min-width: ".concat(minBreakpointSize, "px)") : false, // If 0, don't render a min-width
  maxBreakpointSize ? "(max-width: ".concat(maxBreakpointSize - 1, "px)") : false // If undefined, don't render a max-width
  ].filter(Boolean).join(' and ');
};

exports.euiBreakpoint = euiBreakpoint;

var useEuiBreakpoint = function useEuiBreakpoint(sizes) {
  var euiTheme = (0, _hooks.useEuiTheme)();
  return euiBreakpoint(euiTheme, sizes);
};

exports.useEuiBreakpoint = useEuiBreakpoint;