import _objectSpread from "@babel/runtime/helpers/objectSpread2";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import chroma from 'chroma-js';
import { checkIsMinContinuity, reversePalette, checkIsMaxContinuity, calculateStop, roundValue, getPaletteStops, getDataMinMax, CUSTOM_PALETTE, DEFAULT_RANGE_TYPE, DEFAULT_COLOR_STEPS, DEFAULT_CONTINUITY } from '../../palettes';
import { toColorStops, sortColorRanges } from './color_ranges/utils';

/**
 * Some name conventions here:
 * * `displayStops` => It's an additional transformation of `stops` into a [0, N] domain for the EUIPaletteDisplay component.
 * * `stops` => final steps used to table coloring. It is a rightShift of the colorStops
 * * `colorStops` => user's color stop inputs.  Used to compute range min.
 *
 * When the user inputs the colorStops, they are designed to be the initial part of the color segment,
 * so the next stops indicate where the previous stop ends.
 * Both table coloring logic and EuiPaletteDisplay format implementation works differently than our current `colorStops`,
 * by having the stop values at the end of each color segment rather than at the beginning: `stops` values are computed by a rightShift of `colorStops`.
 * EuiPaletteDisplay has an additional requirement as it is always mapped against a domain [0, N]: from `stops` the `displayStops` are computed with
 * some continuity enrichment and a remap against a [0, 100] domain to make the palette component work ok.
 *
 * These naming conventions would be useful to track the code flow in this feature as multiple transformations are happening
 * for a single change.
 */
export function updateRangeType(newRangeType, activePalette, dataBounds, palettes, colorRanges) {
  var _activePalette$params, _activePalette$params2, _activePalette$params3, _activePalette$params4;

  var continuity = (_activePalette$params = (_activePalette$params2 = activePalette.params) === null || _activePalette$params2 === void 0 ? void 0 : _activePalette$params2.continuity) !== null && _activePalette$params !== void 0 ? _activePalette$params : DEFAULT_CONTINUITY;
  var params = {
    rangeType: newRangeType
  };

  var _getDataMinMax = getDataMinMax(newRangeType, dataBounds),
      newMin = _getDataMinMax.min,
      newMax = _getDataMinMax.max;

  var _getDataMinMax2 = getDataMinMax((_activePalette$params3 = activePalette.params) === null || _activePalette$params3 === void 0 ? void 0 : _activePalette$params3.rangeType, dataBounds),
      oldMin = _getDataMinMax2.min,
      oldMax = _getDataMinMax2.max;

  var newColorStops = getStopsFromColorRangesByNewInterval(colorRanges, {
    oldInterval: oldMax - oldMin,
    newInterval: newMax - newMin,
    newMin: newMin,
    oldMin: oldMin
  });

  if (activePalette.name === CUSTOM_PALETTE) {
    var stops = getPaletteStops(palettes, _objectSpread(_objectSpread({}, activePalette.params), {}, {
      colorStops: newColorStops
    }, params), {
      dataBounds: dataBounds
    });
    params.colorStops = newColorStops;
    params.stops = stops;
  } else {
    params.stops = getPaletteStops(palettes, _objectSpread(_objectSpread({}, activePalette.params), params), {
      prevPalette: activePalette.name,
      dataBounds: dataBounds
    });
  }

  var lastStop = activePalette.name === CUSTOM_PALETTE ? newColorStops[newColorStops.length - 1].stop : params.stops[params.stops.length - 1].stop;
  params.rangeMin = checkIsMinContinuity(continuity) ? Number.NEGATIVE_INFINITY : activePalette.name === CUSTOM_PALETTE ? newColorStops[0].stop : params.stops[0].stop;
  params.rangeMax = checkIsMaxContinuity(continuity) ? Number.POSITIVE_INFINITY : (_activePalette$params4 = activePalette.params) !== null && _activePalette$params4 !== void 0 && _activePalette$params4.rangeMax ? calculateStop(activePalette.params.rangeMax, newMin, oldMin, oldMax - oldMin, newMax - newMin) : lastStop > newMax ? lastStop + 1 : newMax;
  return params;
}
export function changeColorPalette(newPalette, activePalette, palettes, dataBounds, disableSwitchingContinuity) {
  var _activePalette$params5, _activePalette$params6, _activePalette$params7;

  var isNewPaletteCustom = newPalette.name === CUSTOM_PALETTE;

  var newParams = _objectSpread(_objectSpread({}, activePalette.params), {}, {
    name: newPalette.name,
    colorStops: undefined,
    continuity: disableSwitchingContinuity ? (_activePalette$params5 = (_activePalette$params6 = activePalette.params) === null || _activePalette$params6 === void 0 ? void 0 : _activePalette$params6.continuity) !== null && _activePalette$params5 !== void 0 ? _activePalette$params5 : DEFAULT_CONTINUITY : DEFAULT_CONTINUITY,
    reverse: false // restore the reverse flag

  }); // we should pass colorStops so that correct calculate new color stops (if there was before) for custom palette


  var newColorStops = getColorStops(palettes, ((_activePalette$params7 = activePalette.params) === null || _activePalette$params7 === void 0 ? void 0 : _activePalette$params7.colorStops) || [], activePalette, dataBounds);

  if (isNewPaletteCustom) {
    newParams.colorStops = newColorStops;
  }

  return _objectSpread(_objectSpread({}, newPalette), {}, {
    params: _objectSpread(_objectSpread({}, newParams), {}, {
      stops: getPaletteStops(palettes, newParams, {
        prevPalette: isNewPaletteCustom || activePalette.name === CUSTOM_PALETTE ? undefined : newPalette.name,
        dataBounds: dataBounds,
        mapFromMinValue: true
      }),
      rangeMin: checkIsMinContinuity(newParams.continuity) ? Number.NEGATIVE_INFINITY : Math.min(dataBounds.min, newColorStops[0].stop),
      rangeMax: checkIsMaxContinuity(newParams.continuity) ? Number.POSITIVE_INFINITY : Math.min(dataBounds.max, newColorStops[newColorStops.length - 1].stop)
    })
  });
}
export function withUpdatingPalette(palettes, activePalette, colorRanges, dataBounds, continuity) {
  var _ref, _activePalette$params8, _params, _params2, _colorStops$;

  var currentContinuity = (_ref = continuity !== null && continuity !== void 0 ? continuity : (_activePalette$params8 = activePalette.params) === null || _activePalette$params8 === void 0 ? void 0 : _activePalette$params8.continuity) !== null && _ref !== void 0 ? _ref : DEFAULT_CONTINUITY;
  var sortedColorRanges = colorRanges;

  if (colorRanges.some(function (value, index) {
    return index !== colorRanges.length - 1 ? value.start > colorRanges[index + 1].start : false;
  })) {
    sortedColorRanges = sortColorRanges(colorRanges);
  }

  var _toColorStops = toColorStops(sortedColorRanges, currentContinuity),
      max = _toColorStops.max,
      colorStops = _toColorStops.colorStops;

  var newPallete = getSwitchToCustomParams(palettes, activePalette, {
    continuity: currentContinuity,
    colorStops: colorStops,
    steps: ((_params = activePalette.params) === null || _params === void 0 ? void 0 : _params.steps) || DEFAULT_COLOR_STEPS,
    reverse: (_params2 = activePalette.params) === null || _params2 === void 0 ? void 0 : _params2.reverse,
    rangeMin: (_colorStops$ = colorStops[0]) === null || _colorStops$ === void 0 ? void 0 : _colorStops$.stop,
    rangeMax: max
  }, dataBounds);
  return {
    activePalette: newPallete,
    colorRanges: colorRanges
  };
}
export function withUpdatingColorRanges(palettes, activePalette, dataBounds) {
  var _activePalette$params9;

  return {
    colorRanges: toColorRanges(palettes, ((_activePalette$params9 = activePalette.params) === null || _activePalette$params9 === void 0 ? void 0 : _activePalette$params9.colorStops) || [], activePalette, dataBounds),
    activePalette: activePalette
  };
} // Utility to remap color stops within new domain

export function getStopsFromColorRangesByNewInterval(colorRanges, _ref2) {
  var newInterval = _ref2.newInterval,
      oldInterval = _ref2.oldInterval,
      newMin = _ref2.newMin,
      oldMin = _ref2.oldMin;
  return (colorRanges || []).map(function (_ref3) {
    var color = _ref3.color,
        start = _ref3.start;
    var stop = calculateStop(start, newMin, oldMin, oldInterval, newInterval);

    if (oldInterval === 0) {
      stop = newInterval + newMin;
    }

    return {
      color: color,
      stop: roundValue(stop)
    };
  });
}
export function mergePaletteParams(activePalette, newParams) {
  return _objectSpread(_objectSpread({}, activePalette), {}, {
    params: _objectSpread(_objectSpread({}, activePalette.params), newParams)
  });
}

function isValidPonyfill(colorString) {
  // we're using an old version of chroma without the valid function
  try {
    chroma(colorString);
    return true;
  } catch (e) {
    return false;
  }
}

export function isValidColor(colorString) {
  // chroma can handle also hex values with alpha channel/transparency
  // chroma accepts also hex without #, so test for it
  return colorString !== '' && /^#/.test(colorString) && isValidPonyfill(colorString);
}

function getSwitchToCustomParams(palettes, activePalette, newParams, dataBounds) {
  var _activePalette$params10;

  // if it's already a custom palette just return the params
  if ((activePalette === null || activePalette === void 0 ? void 0 : (_activePalette$params10 = activePalette.params) === null || _activePalette$params10 === void 0 ? void 0 : _activePalette$params10.name) === CUSTOM_PALETTE) {
    var _stops = getPaletteStops(palettes, _objectSpread(_objectSpread({
      steps: DEFAULT_COLOR_STEPS
    }, activePalette.params), newParams), {
      dataBounds: dataBounds
    });

    return mergePaletteParams(activePalette, _objectSpread(_objectSpread({}, newParams), {}, {
      stops: _stops
    }));
  } // prepare everything to switch to custom palette


  var newPaletteParams = _objectSpread(_objectSpread(_objectSpread({
    steps: DEFAULT_COLOR_STEPS
  }, activePalette.params), newParams), {}, {
    name: CUSTOM_PALETTE
  });

  var stops = getPaletteStops(palettes, newPaletteParams, {
    prevPalette: newPaletteParams.colorStops ? undefined : activePalette.name,
    dataBounds: dataBounds
  });
  return mergePaletteParams({
    name: CUSTOM_PALETTE,
    type: 'palette'
  }, _objectSpread(_objectSpread({}, newPaletteParams), {}, {
    stops: stops
  }));
}

export function getColorStops(palettes, colorStops, activePalette, dataBounds) {
  var _activePalette$params11;

  // just forward the current stops if custom
  if ((activePalette === null || activePalette === void 0 ? void 0 : activePalette.name) === CUSTOM_PALETTE && colorStops !== null && colorStops !== void 0 && colorStops.length) {
    return colorStops;
  } // for predefined palettes create some stops, then drop the last one.
  // we're using these as starting point for the user


  var freshColorStops = getPaletteStops(palettes, _objectSpread({}, activePalette === null || activePalette === void 0 ? void 0 : activePalette.params), // mapFromMinValue is a special flag to offset the stops values
  // used here to avoid a new remap/left shift
  {
    dataBounds: dataBounds,
    mapFromMinValue: true,
    defaultPaletteName: activePalette.name
  });

  if (activePalette !== null && activePalette !== void 0 && (_activePalette$params11 = activePalette.params) !== null && _activePalette$params11 !== void 0 && _activePalette$params11.reverse) {
    freshColorStops = reversePalette(freshColorStops);
  }

  return freshColorStops;
}
/**
 * Both table coloring logic and EuiPaletteDisplay format implementation works differently than our current `colorStops`,
 * by having the stop values at the end of each color segment rather than at the beginning: `stops` values are computed by a rightShift of `colorStops`.
 * EuiPaletteDisplay has an additional requirement as it is always mapped against a domain [0, N]: from `stops` the `displayStops` are computed with
 * some continuity enrichment and a remap against a [0, 100] domain to make the palette component work ok.
 *
 * These naming conventions would be useful to track the code flow in this feature as multiple transformations are happening
 * for a single change.
 */

export function toColorRanges(palettes, colorStops, activePalette, dataBounds) {
  var _activePalette$params12;

  var _ref4 = (_activePalette$params12 = activePalette.params) !== null && _activePalette$params12 !== void 0 ? _activePalette$params12 : {},
      _ref4$continuity = _ref4.continuity,
      continuity = _ref4$continuity === void 0 ? DEFAULT_CONTINUITY : _ref4$continuity,
      _ref4$rangeType = _ref4.rangeType,
      rangeType = _ref4$rangeType === void 0 ? DEFAULT_RANGE_TYPE : _ref4$rangeType;

  var _getDataMinMax3 = getDataMinMax(rangeType, dataBounds),
      dataMin = _getDataMinMax3.min,
      dataMax = _getDataMinMax3.max;

  return getColorStops(palettes, colorStops || [], activePalette, dataBounds).map(function (colorStop, index, array) {
    var _ref5, _colorStop$stop, _activePalette$params13, _ref6, _array$stop, _array, _activePalette$params14;

    var isFirst = index === 0;
    var isLast = index === array.length - 1;
    return {
      color: colorStop.color,
      start: isFirst && checkIsMinContinuity(continuity) ? Number.NEGATIVE_INFINITY : (_ref5 = (_colorStop$stop = colorStop.stop) !== null && _colorStop$stop !== void 0 ? _colorStop$stop : (_activePalette$params13 = activePalette.params) === null || _activePalette$params13 === void 0 ? void 0 : _activePalette$params13.rangeMin) !== null && _ref5 !== void 0 ? _ref5 : dataMin,
      end: isLast && checkIsMaxContinuity(continuity) ? Number.POSITIVE_INFINITY : (_ref6 = (_array$stop = (_array = array[index + 1]) === null || _array === void 0 ? void 0 : _array.stop) !== null && _array$stop !== void 0 ? _array$stop : (_activePalette$params14 = activePalette.params) === null || _activePalette$params14 === void 0 ? void 0 : _activePalette$params14.rangeMax) !== null && _ref6 !== void 0 ? _ref6 : dataMax
    };
  });
}