"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentManager = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _http = require("http");

var _https = require("https");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const HTTPS = 'https:';
const DEFAULT_CONFIG = {
  keepAlive: true,
  keepAliveMsecs: 1000,
  maxSockets: 256,
  maxFreeSockets: 256,
  scheduling: 'lifo'
};

/**
 * Allows obtaining Agent factories, which can then be fed into elasticsearch-js's Client class.
 * Ideally, we should obtain one Agent factory for each ES Client class.
 * This allows using the same Agent across all the Pools and Connections of the Client (one per ES node).
 *
 * Agent instances are stored internally to allow collecting metrics (nbr of active/idle connections to ES).
 *
 * Using the same Agent factory across multiple ES Client instances is strongly discouraged, cause ES Client
 * exposes methods that can modify the underlying pools, effectively impacting the connections of other Clients.
 * @internal
 **/
class AgentManager {
  // Stores Https Agent instances
  // Stores Http Agent instances
  constructor(agentOptions = DEFAULT_CONFIG) {
    (0, _defineProperty2.default)(this, "httpsStore", void 0);
    (0, _defineProperty2.default)(this, "httpStore", void 0);
    this.agentOptions = agentOptions;
    this.httpsStore = new Set();
    this.httpStore = new Set();
  }

  getAgentFactory(agentOptions) {
    // a given agent factory always provides the same Agent instances (for the same protocol)
    // we keep references to the instances at factory level, to be able to reuse them
    let httpAgent;
    let httpsAgent;
    return connectionOpts => {
      if (connectionOpts.url.protocol === HTTPS) {
        if (!httpsAgent) {
          const config = Object.assign({}, DEFAULT_CONFIG, this.agentOptions, agentOptions, connectionOpts.tls);
          httpsAgent = new _https.Agent(config);
          this.httpsStore.add(httpsAgent);
          dereferenceOnDestroy(this.httpsStore, httpsAgent);
        }

        return httpsAgent;
      }

      if (!httpAgent) {
        const config = Object.assign({}, DEFAULT_CONFIG, this.agentOptions, agentOptions);
        httpAgent = new _http.Agent(config);
        this.httpStore.add(httpAgent);
        dereferenceOnDestroy(this.httpStore, httpAgent);
      }

      return httpAgent;
    };
  }

}

exports.AgentManager = AgentManager;

const dereferenceOnDestroy = (protocolStore, agent) => {
  const doDestroy = agent.destroy.bind(agent);

  agent.destroy = () => {
    protocolStore.delete(agent);
    doDestroy();
  };
};