"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addExcludedTypesToBoolQuery = addExcludedTypesToBoolQuery;
exports.addMustClausesToBoolQuery = addMustClausesToBoolQuery;
exports.addMustNotClausesToBoolQuery = addMustNotClausesToBoolQuery;
exports.getAliases = getAliases;
exports.indexBelongsToLaterVersion = indexBelongsToLaterVersion;
exports.indexVersion = indexVersion;
exports.mergeMigrationMappingPropertyHashes = mergeMigrationMappingPropertyHashes;
exports.throwBadControlState = throwBadControlState;
exports.throwBadResponse = throwBadResponse;

var _semver = require("semver");

var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
function throwBadControlState(controlState) {
  throw new Error('Unexpected control state: ' + controlState);
}
/**
 * A helper function/type for ensuring that all response types are handled.
 */


function throwBadResponse(state, res) {
  throw new Error(`${state.controlState} received unexpected action response: ` + JSON.stringify(res));
}
/**
 * Merge the _meta.migrationMappingPropertyHashes mappings of an index with
 * the given target mappings.
 *
 * @remarks When another instance already completed a migration, the existing
 * target index might contain documents and mappings created by a plugin that
 * is disabled in the current Kibana instance performing this migration.
 * Mapping updates are commutative (deeply merged) by Elasticsearch, except
 * for the `_meta` key. By merging the `_meta.migrationMappingPropertyHashes`
 * mappings from the existing target index index into the targetMappings we
 * ensure that any `migrationPropertyHashes` for disabled plugins aren't lost.
 *
 * Right now we don't use these `migrationPropertyHashes` but it could be used
 * in the future to detect if mappings were changed. If mappings weren't
 * changed we don't need to reindex but can clone the index to save disk space.
 *
 * @param targetMappings
 * @param indexMappings
 */


function mergeMigrationMappingPropertyHashes(targetMappings, indexMappings) {
  var _indexMappings$_meta, _targetMappings$_meta;

  return { ...targetMappings,
    _meta: {
      migrationMappingPropertyHashes: { ...((_indexMappings$_meta = indexMappings._meta) === null || _indexMappings$_meta === void 0 ? void 0 : _indexMappings$_meta.migrationMappingPropertyHashes),
        ...((_targetMappings$_meta = targetMappings._meta) === null || _targetMappings$_meta === void 0 ? void 0 : _targetMappings$_meta.migrationMappingPropertyHashes)
      }
    }
  };
}

function indexBelongsToLaterVersion(indexName, kibanaVersion) {
  const version = (0, _semver.valid)(indexVersion(indexName));
  return version != null ? (0, _semver.gt)(version, kibanaVersion) : false;
}
/**
 * Add new must_not clauses to the given query
 * in order to filter out the specified types
 * @param boolQuery the bool query to be enriched
 * @param types the types to be filtered out
 * @returns a new query container with the enriched query
 */


function addExcludedTypesToBoolQuery(types, boolQuery) {
  return addMustNotClausesToBoolQuery(types.map(type => ({
    term: {
      type
    }
  })), boolQuery);
}
/**
 * Add the given clauses to the 'must' of the given query
 * @param boolQuery the bool query to be enriched
 * @param mustClauses the clauses to be added to a 'must'
 * @returns a new query container with the enriched query
 */


function addMustClausesToBoolQuery(mustClauses, boolQuery) {
  let must = [];

  if (boolQuery !== null && boolQuery !== void 0 && boolQuery.must) {
    must = must.concat(boolQuery.must);
  }

  must.push(...mustClauses);
  return {
    bool: { ...boolQuery,
      must
    }
  };
}
/**
 * Add the given clauses to the 'must_not' of the given query
 * @param boolQuery the bool query to be enriched
 * @param mustNotClauses the clauses to be added to a 'must_not'
 * @returns a new query container with the enriched query
 */


function addMustNotClausesToBoolQuery(mustNotClauses, boolQuery) {
  let mustNot = [];

  if (boolQuery !== null && boolQuery !== void 0 && boolQuery.must_not) {
    mustNot = mustNot.concat(boolQuery.must_not);
  }

  mustNot.push(...mustNotClauses);
  return {
    bool: { ...boolQuery,
      must_not: mustNot
    }
  };
}
/**
 * Extracts the version number from a >= 7.11 index
 * @param indexName A >= v7.11 index name
 */


function indexVersion(indexName) {
  return ((indexName === null || indexName === void 0 ? void 0 : indexName.match(/.+_(\d+\.\d+\.\d+)_\d+/)) || [])[1];
}
/**
 * Creates a record of alias -> index name pairs
 */


function getAliases(indices) {
  const aliases = {};

  for (const index of Object.getOwnPropertyNames(indices)) {
    for (const alias of Object.getOwnPropertyNames(indices[index].aliases || {})) {
      if (aliases[alias] != null) {
        return Either.left({
          type: 'multiple_indices_per_alias',
          alias,
          indices: [aliases[alias], index]
        });
      }

      aliases[alias] = index;
    }
  }

  return Either.right(aliases);
}