"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MIGRATION_CLIENT_OPTIONS = void 0;
exports.runResilientMigrator = runResilientMigrator;

var _next = require("./next");

var _model = require("./model");

var _initial_state = require("./initial_state");

var _migrations_state_action_machine = require("./migrations_state_action_machine");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * To avoid the Elasticsearch-js client aborting our requests before we
 * receive a response from Elasticsearch we choose a requestTimeout that's
 * longer than the DEFAULT_TIMEOUT.
 *
 * This timeout is only really valuable for preventing migrations from being
 * stuck waiting forever for a response when the underlying socket is broken.
 *
 * We also set maxRetries to 0 so that the state action machine can handle all
 * retries. This way we get exponential back-off and logging for failed
 * actions.
 */
const MIGRATION_CLIENT_OPTIONS = {
  maxRetries: 0,
  requestTimeout: 120_000
};
/**
 * Migrates the provided indexPrefix index using a resilient algorithm that is
 * completely lock-free so that any failure can always be retried by
 * restarting Kibana.
 */

exports.MIGRATION_CLIENT_OPTIONS = MIGRATION_CLIENT_OPTIONS;

async function runResilientMigrator({
  client,
  kibanaVersion,
  targetMappings,
  logger,
  preMigrationScript,
  transformRawDocs,
  migrationVersionPerType,
  indexPrefix,
  migrationsConfig,
  typeRegistry,
  docLinks
}) {
  const initialState = (0, _initial_state.createInitialState)({
    kibanaVersion,
    targetMappings,
    preMigrationScript,
    migrationVersionPerType,
    indexPrefix,
    migrationsConfig,
    typeRegistry,
    docLinks,
    logger
  });
  const migrationClient = client.child(MIGRATION_CLIENT_OPTIONS);
  return (0, _migrations_state_action_machine.migrationStateActionMachine)({
    initialState,
    logger,
    next: (0, _next.next)(migrationClient, transformRawDocs),
    model: _model.model,
    client: migrationClient
  });
}