import _defineProperty from "@babel/runtime/helpers/defineProperty";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { get, has, isPlainObject } from 'lodash';
import { getConvertedValueForField } from './get_converted_value_for_field';
import { hasRangeKeys } from './range_filter';

/**
 * @param filter
 * @returns `true` if a filter is a `PhraseFilter`
 *
 * @public
 */
export var isPhraseFilter = function isPhraseFilter(filter) {
  var isMatchPhraseQuery = has(filter, 'query.match_phrase');
  var matchQueryPart = get(filter, 'query.match', []);
  var isDeprecatedMatchPhraseQuery = Object.values(matchQueryPart).find(function (params) {
    return params.type === 'phrase';
  }) !== undefined;
  return isMatchPhraseQuery || isDeprecatedMatchPhraseQuery;
};
/**
 * @param filter
 * @returns `true` if a filter is a scripted `PhrasesFilter`
 *
 * @public
 */

export var isScriptedPhraseFilter = function isScriptedPhraseFilter(filter) {
  var _filter$query, _filter$query$script, _filter$query$script$;

  return has(filter, 'query.script.script.params.value') && !hasRangeKeys((_filter$query = filter.query) === null || _filter$query === void 0 ? void 0 : (_filter$query$script = _filter$query.script) === null || _filter$query$script === void 0 ? void 0 : (_filter$query$script$ = _filter$query$script.script) === null || _filter$query$script$ === void 0 ? void 0 : _filter$query$script$.params);
};
/** @internal */

export var getPhraseFilterField = function getPhraseFilterField(filter) {
  var _filter$meta, _ref, _filter$query$match_p;

  if ((_filter$meta = filter.meta) !== null && _filter$meta !== void 0 && _filter$meta.field) return filter.meta.field;
  var queryConfig = (_ref = (_filter$query$match_p = filter.query.match_phrase) !== null && _filter$query$match_p !== void 0 ? _filter$query$match_p : filter.query.match) !== null && _ref !== void 0 ? _ref : {};
  return Object.keys(queryConfig)[0];
};
/**
 * @internal
 */

export var getPhraseFilterValue = function getPhraseFilterValue(filter) {
  if (isPhraseFilter(filter)) {
    var queryConfig = filter.query.match_phrase || filter.query.match || {};
    var queryValue = Object.values(queryConfig)[0];
    return isPlainObject(queryValue) ? queryValue.query : queryValue;
  } else {
    var _filter$query2, _filter$query2$script, _filter$query2$script2, _filter$query2$script3;

    return (_filter$query2 = filter.query) === null || _filter$query2 === void 0 ? void 0 : (_filter$query2$script = _filter$query2.script) === null || _filter$query2$script === void 0 ? void 0 : (_filter$query2$script2 = _filter$query2$script.script) === null || _filter$query2$script2 === void 0 ? void 0 : (_filter$query2$script3 = _filter$query2$script2.params) === null || _filter$query2$script3 === void 0 ? void 0 : _filter$query2$script3.value;
  }
};
/**
 * Creates a filter where the given field matches a given value
 * @param field
 * @param params
 * @param indexPattern
 * @returns `PhraseFilter`
 *
 * @public
 */

export var buildPhraseFilter = function buildPhraseFilter(field, value, indexPattern) {
  var convertedValue = getConvertedValueForField(field, value);

  if (field.scripted) {
    return {
      meta: {
        index: indexPattern.id,
        field: field.name
      },
      query: {
        script: getPhraseScript(field, value)
      }
    };
  } else {
    return {
      meta: {
        index: indexPattern.id
      },
      query: {
        match_phrase: _defineProperty({}, field.name, convertedValue)
      }
    };
  }
};
/** @internal */

export var getPhraseScript = function getPhraseScript(field, value) {
  var convertedValue = getConvertedValueForField(field, value);
  var script = buildInlineScriptForPhraseFilter(field);
  return {
    script: {
      source: script,
      lang: field.lang,
      params: {
        value: convertedValue
      }
    }
  };
};
/**
 * @internal
 * Takes a scripted field and returns an inline script appropriate for use in a script query.
 * Handles lucene expression and Painless scripts. Other langs aren't guaranteed to generate valid
 * scripts.
 *
 * @param {object} scriptedField A Field object representing a scripted field
 * @returns {string} The inline script string
 */

export var buildInlineScriptForPhraseFilter = function buildInlineScriptForPhraseFilter(scriptedField) {
  // We must wrap painless scripts in a lambda in case they're more than a simple expression
  if (scriptedField.lang === 'painless') {
    return "boolean compare(Supplier s, def v) {return s.get() == v;}" + "compare(() -> { ".concat(scriptedField.script, " }, params.value);");
  } else {
    return "(".concat(scriptedField.script, ") == value");
  }
};