import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { isUndefined } from 'lodash';
import { getPhraseScript } from '../../filters';
import { getFields } from './utils/get_fields';
import { getTimeZoneFromSettings, getDataViewFieldSubtypeNested } from '../../utils';
import { getFullFieldNameNode } from './utils/get_full_field_name_node';
import * as ast from '../ast';
import * as literal from '../node_types/literal';
import * as wildcard from '../node_types/wildcard';
export function buildNodeParams(fieldName, value) {
  if (isUndefined(fieldName)) {
    throw new Error('fieldName is a required argument');
  }

  if (isUndefined(value)) {
    throw new Error('value is a required argument');
  }

  var fieldNode = typeof fieldName === 'string' ? ast.fromLiteralExpression(fieldName) : literal.buildNode(fieldName);
  var valueNode = typeof value === 'string' ? ast.fromLiteralExpression(value) : literal.buildNode(value);
  return {
    arguments: [fieldNode, valueNode]
  };
}
export function toElasticsearchQuery(node, indexPattern) {
  var config = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var context = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};

  var _node$arguments = _slicedToArray(node.arguments, 2),
      fieldNameArg = _node$arguments[0],
      valueArg = _node$arguments[1];

  var isExistsQuery = wildcard.isNode(valueArg) && wildcard.isLoneWildcard(valueArg);
  var isAllFieldsQuery = wildcard.isNode(fieldNameArg) && wildcard.isLoneWildcard(fieldNameArg);
  var isMatchAllQuery = isExistsQuery && isAllFieldsQuery;

  if (isMatchAllQuery) {
    return {
      match_all: {}
    };
  }

  var fullFieldNameArg = getFullFieldNameNode(fieldNameArg, indexPattern, context !== null && context !== void 0 && context.nested ? context.nested.path : undefined);
  var value = !isUndefined(valueArg) ? ast.toElasticsearchQuery(valueArg) : valueArg;
  var type = valueArg.isQuoted ? 'phrase' : 'best_fields';

  if (fullFieldNameArg.value === null) {
    if (wildcard.isNode(valueArg)) {
      return {
        query_string: {
          query: wildcard.toQueryStringQuery(valueArg)
        }
      };
    }

    return {
      multi_match: {
        type: type,
        query: value,
        lenient: true
      }
    };
  }

  var fields = indexPattern ? getFields(fullFieldNameArg, indexPattern) : []; // If no fields are found in the index pattern we send through the given field name as-is. We do this to preserve
  // the behaviour of lucene on dashboards where there are panels based on different index patterns that have different
  // fields. If a user queries on a field that exists in one pattern but not the other, the index pattern without the
  // field should return no results. It's debatable whether this is desirable, but it's been that way forever, so we'll
  // keep things familiar for now.

  if (fields && fields.length === 0) {
    fields.push({
      name: ast.toElasticsearchQuery(fullFieldNameArg),
      scripted: false,
      type: ''
    });
  } // Special case for wildcards where there are no fields or all fields share the same prefix


  if (isExistsQuery && (!(fields !== null && fields !== void 0 && fields.length) || (fields === null || fields === void 0 ? void 0 : fields.length) === (indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.fields.length))) {
    return {
      match_all: {}
    };
  }

  var queries = fields.reduce(function (accumulator, field) {
    var wrapWithNestedQuery = function wrapWithNestedQuery(query) {
      // Wildcards can easily include nested and non-nested fields. There isn't a good way to let
      // users handle this themselves so we automatically add nested queries in this scenario.
      var subTypeNested = getDataViewFieldSubtypeNested(field);

      if (!wildcard.isNode(fullFieldNameArg) || !(subTypeNested !== null && subTypeNested !== void 0 && subTypeNested.nested) || context !== null && context !== void 0 && context.nested) {
        return query;
      } else {
        return {
          nested: _objectSpread({
            path: subTypeNested.nested.path,
            query: query,
            score_mode: 'none'
          }, typeof config.nestedIgnoreUnmapped === 'boolean' && {
            ignore_unmapped: config.nestedIgnoreUnmapped
          })
        };
      }
    };

    if (field.scripted) {
      // Exists queries don't make sense for scripted fields
      if (!isExistsQuery) {
        return [].concat(_toConsumableArray(accumulator), [{
          script: _objectSpread({}, getPhraseScript(field, value))
        }]);
      }
    } else if (isExistsQuery) {
      return [].concat(_toConsumableArray(accumulator), [wrapWithNestedQuery({
        exists: {
          field: field.name
        }
      })]);
    } else if (wildcard.isNode(valueArg)) {
      var _field$esTypes;

      var query = (_field$esTypes = field.esTypes) !== null && _field$esTypes !== void 0 && _field$esTypes.includes('keyword') ? {
        wildcard: _defineProperty({}, field.name, value)
      } : {
        query_string: {
          fields: [field.name],
          query: wildcard.toQueryStringQuery(valueArg)
        }
      };
      return [].concat(_toConsumableArray(accumulator), [wrapWithNestedQuery(query)]);
    } else if (field.type === 'date') {
      /*
      If we detect that it's a date field and the user wants an exact date, we need to convert the query to both >= and <= the value provided to force a range query. This is because match and match_phrase queries do not accept a timezone parameter.
      dateFormatTZ can have the value of 'Browser', in which case we guess the timezone using moment.tz.guess.
      */
      var timeZoneParam = config.dateFormatTZ ? {
        time_zone: getTimeZoneFromSettings(config.dateFormatTZ)
      } : {};
      return [].concat(_toConsumableArray(accumulator), [wrapWithNestedQuery({
        range: _defineProperty({}, field.name, _objectSpread({
          gte: value,
          lte: value
        }, timeZoneParam))
      })]);
    } else {
      var queryType = type === 'phrase' ? 'match_phrase' : 'match';
      return [].concat(_toConsumableArray(accumulator), [wrapWithNestedQuery(_defineProperty({}, queryType, _defineProperty({}, field.name, value)))]);
    }
  }, []);
  return {
    bool: {
      should: queries || [],
      minimum_should_match: 1
    }
  };
}