/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
export var KQL_WILDCARD_SYMBOL = '@kuery-wildcard@';
export var KQL_NODE_TYPE_WILDCARD = 'wildcard';

// Copied from https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions
function escapeRegExp(str) {
  return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
} // See https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#_reserved_characters


function escapeQueryString(str) {
  return str.replace(/[+-=&|><!(){}[\]^"~*?:\\/]/g, '\\$&'); // $& means the whole matched string
}

export function isNode(node) {
  return node.type === KQL_NODE_TYPE_WILDCARD;
}
export function isMatchAll(node) {
  return node.value === KQL_WILDCARD_SYMBOL;
}
export function buildNode(value) {
  // When called from the parser, all wildcard characters are replaced with a special flag (since escaped wildcards are
  // handled as normal strings). However, when invoking programmatically, callers shouldn't need to do this replacement.
  if (!value.includes(KQL_NODE_TYPE_WILDCARD) && value.includes('*')) {
    return buildNode(value.replaceAll('*', KQL_WILDCARD_SYMBOL));
  }

  return {
    type: KQL_NODE_TYPE_WILDCARD,
    value: value
  };
}
export function test(node, str) {
  var value = node.value;
  var regex = value.split(KQL_WILDCARD_SYMBOL).map(escapeRegExp).join('[\\s\\S]*');
  var regexp = new RegExp("^".concat(regex, "$"));
  return regexp.test(str);
}
export function toElasticsearchQuery(node) {
  var value = node.value;
  return value.split(KQL_WILDCARD_SYMBOL).join('*');
}
export function toQueryStringQuery(node) {
  var value = node.value;
  return value.split(KQL_WILDCARD_SYMBOL).map(escapeQueryString).join('*');
}
export function isLoneWildcard(_ref) {
  var value = _ref.value;
  return value.includes(KQL_WILDCARD_SYMBOL) && value.replace(KQL_WILDCARD_SYMBOL, '').length === 0;
}
export function hasLeadingWildcard(node) {
  return !isLoneWildcard(node) && node.value.startsWith(KQL_WILDCARD_SYMBOL);
}