/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { AggregationTemporality, InstrumentType } from '@opentelemetry/sdk-metrics-base';
import { defaultOptions } from './OTLPMetricExporterOptions';
export const CumulativeTemporalitySelector = () => AggregationTemporality.CUMULATIVE;
export const DeltaTemporalitySelector = (instrumentType) => {
    switch (instrumentType) {
        case InstrumentType.COUNTER:
        case InstrumentType.OBSERVABLE_COUNTER:
        case InstrumentType.HISTOGRAM:
        case InstrumentType.OBSERVABLE_GAUGE:
            return AggregationTemporality.DELTA;
        case InstrumentType.UP_DOWN_COUNTER:
        case InstrumentType.OBSERVABLE_UP_DOWN_COUNTER:
            return AggregationTemporality.CUMULATIVE;
    }
};
function chooseTemporalitySelector(temporalityPreference) {
    if (temporalityPreference === AggregationTemporality.DELTA) {
        return DeltaTemporalitySelector;
    }
    return CumulativeTemporalitySelector;
}
export class OTLPMetricExporterBase {
    constructor(exporter, config = defaultOptions) {
        this._otlpExporter = exporter;
        this._aggregationTemporalitySelector = chooseTemporalitySelector(config.temporalityPreference);
    }
    export(metrics, resultCallback) {
        this._otlpExporter.export([metrics], resultCallback);
    }
    async shutdown() {
        await this._otlpExporter.shutdown();
    }
    forceFlush() {
        return Promise.resolve();
    }
    selectAggregationTemporality(instrumentType) {
        return this._aggregationTemporalitySelector(instrumentType);
    }
}
//# sourceMappingURL=OTLPMetricExporterBase.js.map