"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dropDuplicates = dropDuplicates;
exports.fetchFrequentItems = fetchFrequentItems;
exports.groupDuplicates = groupDuplicates;

var _lodash = require("lodash");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function dropDuplicates(cps, uniqueFields) {
  return (0, _lodash.uniqWith)(cps, (a, b) => (0, _lodash.isEqual)((0, _lodash.pick)(a, uniqueFields), (0, _lodash.pick)(b, uniqueFields)));
}

function groupDuplicates(cps, uniqueFields) {
  const groups = [];

  for (const cp of cps) {
    const compareAttributes = (0, _lodash.pick)(cp, uniqueFields);
    const groupIndex = groups.findIndex(g => (0, _lodash.isEqual)(g.keys, compareAttributes));

    if (groupIndex === -1) {
      groups.push({
        keys: compareAttributes,
        group: [cp]
      });
    } else {
      groups[groupIndex].group.push(cp);
    }
  }

  return groups;
}

async function fetchFrequentItems(client, index, searchQuery, changePoints, timeFieldName, deviationMin, deviationMax, logger, emitError) {
  // get unique fields from change points
  const fields = [...new Set(changePoints.map(t => t.fieldName))]; // TODO add query params

  const query = {
    bool: {
      minimum_should_match: 2,
      filter: [searchQuery, {
        range: {
          [timeFieldName]: {
            gte: deviationMin,
            lt: deviationMax
          }
        }
      }],
      should: changePoints.map(t => {
        return {
          term: {
            [t.fieldName]: t.fieldValue
          }
        };
      })
    }
  };
  const aggFields = fields.map(field => ({
    field
  }));
  const totalDocCount = changePoints[0].total_doc_count;
  const minDocCount = 50000;
  let sampleProbability = 1;

  if (totalDocCount > minDocCount) {
    sampleProbability = Math.min(0.5, minDocCount / totalDocCount);
  }

  logger.debug(`frequent_items sample probability: ${sampleProbability}`); // frequent items can be slow, so sample and use 10% min_support

  const aggs = {
    sample: {
      random_sampler: {
        probability: sampleProbability
      },
      aggs: {
        fi: {
          // @ts-expect-error `frequent_items` is not yet part of `AggregationsAggregationContainer`
          frequent_items: {
            minimum_set_size: 2,
            size: 200,
            minimum_support: 0.1,
            fields: aggFields
          }
        }
      }
    }
  };
  const body = await client.search({
    index,
    size: 0,
    body: {
      query,
      aggs,
      size: 0,
      track_total_hits: true
    }
  }, {
    maxRetries: 0
  });

  if (body.aggregations === undefined) {
    logger.error(`Failed to fetch frequent_items, got: \n${JSON.stringify(body, null, 2)}`);
    emitError(`Failed to fetch frequent_items.`);
    return {
      fields: [],
      df: [],
      totalDocCount: 0
    };
  }

  const totalDocCountFi = body.hits.total.value;
  const shape = body.aggregations.sample.fi.buckets.length;
  let maximum = shape;

  if (maximum > 50000) {
    maximum = 50000;
  }

  const fiss = body.aggregations.sample.fi.buckets;
  fiss.length = maximum;
  const results = [];
  fiss.forEach(fis => {
    const result = {
      set: {},
      size: 0,
      maxPValue: 0,
      doc_count: 0,
      support: 0,
      total_doc_count: 0
    };
    let maxPValue;
    Object.entries(fis.key).forEach(([key, value]) => {
      var _changePoints$find;

      result.set[key] = value[0];
      const pValue = (_changePoints$find = changePoints.find(t => t.fieldName === key && t.fieldValue === value[0])) === null || _changePoints$find === void 0 ? void 0 : _changePoints$find.pValue;

      if (pValue !== undefined && pValue !== null) {
        var _maxPValue;

        maxPValue = Math.max((_maxPValue = maxPValue) !== null && _maxPValue !== void 0 ? _maxPValue : 0, pValue);
      }
    });

    if (maxPValue === undefined) {
      return;
    }

    result.size = Object.keys(result.set).length;
    result.maxPValue = maxPValue;
    result.doc_count = fis.doc_count;
    result.support = fis.support;
    result.total_doc_count = totalDocCountFi;
    results.push(result);
  });
  results.sort((a, b) => {
    return b.doc_count - a.doc_count;
  });
  const uniqueFields = (0, _lodash.uniq)(results.flatMap(r => Object.keys(r.set)));
  return {
    fields: uniqueFields,
    df: results,
    totalDocCount: totalDocCountFi
  };
}