"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServerlessFunctionLatency = getServerlessFunctionLatency;

var _i18n = require("@kbn/i18n");

var _uiTheme = require("@kbn/ui-theme");

var _elasticsearch_fieldnames = require("../../../../../common/elasticsearch_fieldnames");

var _latency_aggregation_types = require("../../../../../common/latency_aggregation_types");

var _is_finite_number = require("../../../../../common/utils/is_finite_number");

var _viz_colors = require("../../../../../common/viz_colors");

var _get_latency_charts = require("../../../transactions/get_latency_charts");

var _fetch_and_transform_metrics = require("../../fetch_and_transform_metrics");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const billedDurationAvg = {
  title: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.billedDurationAvg', {
    defaultMessage: 'Billed Duration'
  })
};
const chartBase = {
  title: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.avgDuration', {
    defaultMessage: 'Avg. Duration'
  }),
  key: 'avg_duration',
  type: 'linemark',
  yUnit: 'time',
  series: {},
  description: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.avgDuration.description', {
    defaultMessage: 'Transaction duration is the time spent processing and responding to a request. If the request is queued it will not be contribute to the transaction duration but will contribute the overall billed duration'
  })
};

async function getServerlessLantecySeries({
  environment,
  kuery,
  setup,
  serviceName,
  start,
  end,
  searchAggregatedTransactions
}) {
  var _transactionLatency$o;

  const transactionLatency = await (0, _get_latency_charts.getLatencyTimeseries)({
    environment,
    kuery,
    serviceName,
    setup,
    searchAggregatedTransactions,
    latencyAggregationType: _latency_aggregation_types.LatencyAggregationType.avg,
    start,
    end
  });
  return [{
    title: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.transactionDuration', {
      defaultMessage: 'Transaction Duration'
    }),
    key: 'transaction_duration',
    type: 'linemark',
    color: (0, _viz_colors.getVizColorForIndex)(1, _uiTheme.euiLightVars),
    overallValue: (_transactionLatency$o = transactionLatency.overallAvgDuration) !== null && _transactionLatency$o !== void 0 ? _transactionLatency$o : 0,
    data: transactionLatency.latencyTimeseries
  }];
}

async function getServerlessFunctionLatency({
  environment,
  kuery,
  setup,
  serviceName,
  start,
  end,
  searchAggregatedTransactions
}) {
  var _series$data;

  const options = {
    environment,
    kuery,
    setup,
    serviceName,
    start,
    end
  };
  const [billedDurationMetrics, serverlessDurationSeries] = await Promise.all([(0, _fetch_and_transform_metrics.fetchAndTransformMetrics)({ ...options,
    chartBase: { ...chartBase,
      series: {
        billedDurationAvg
      }
    },
    aggs: {
      billedDurationAvg: {
        avg: {
          field: _elasticsearch_fieldnames.FAAS_BILLED_DURATION
        }
      }
    },
    additionalFilters: [{
      exists: {
        field: _elasticsearch_fieldnames.FAAS_BILLED_DURATION
      }
    }],
    operationName: 'get_billed_duration'
  }), getServerlessLantecySeries({ ...options,
    searchAggregatedTransactions
  })]);
  const [series] = billedDurationMetrics.series;
  const data = series === null || series === void 0 ? void 0 : (_series$data = series.data) === null || _series$data === void 0 ? void 0 : _series$data.map(({
    x,
    y
  }) => ({
    x,
    // Billed duration is stored in ms, convert it to microseconds so it uses the same unit as the other chart
    y: (0, _is_finite_number.isFiniteNumber)(y) ? y * 1000 : y
  }));
  return { ...billedDurationMetrics,
    series: series ? [{ ...series,
      // Billed duration is stored in ms, convert it to microseconds
      overallValue: series.overallValue * 1000,
      data
    }, ...serverlessDurationSeries] : []
  };
}