"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRulesConfigurationHandler = exports.updatePackagePolicyRuntimeCfgVar = exports.updatePackagePolicyCspRules = exports.getUpdatedPackagePolicy = exports.getCspRulesSO = exports.defineUpdateRulesConfigRoute = exports.createRulesConfig = void 0;

var _configSchema = require("@kbn/config-schema");

var _jsYaml = _interopRequireDefault(require("js-yaml"));

var _common = require("../../../../fleet/common");

var _server = require("../../../../../../src/core/server");

var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");

var _lodash = require("lodash");

var _immer = _interopRequireDefault(require("immer"));

var _helpers = require("../../../common/utils/helpers");

var _constants = require("../../../common/constants");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const RUNTIME_CFG_FIELDS = ['enabled', 'metadata.benchmark.id', 'metadata.rego_rule_id'];

const updateRulesConfigurationBodySchema = _configSchema.schema.object({
  /**
   * CSP integration instance ID
   */
  package_policy_id: _configSchema.schema.string(),

  /**
   * CSP rules to update
   */
  rules: _configSchema.schema.arrayOf(_configSchema.schema.object({
    /**
     * the rule saved-object id
     */
    id: _configSchema.schema.string(),
    enabled: _configSchema.schema.boolean()
  }))
});

const getEnabledRulesByBenchmark = rules => rules.reduce((benchmarks, rule) => {
  const benchmark = rule.attributes.metadata.benchmark.id;
  if (!rule.attributes.enabled || !benchmark) return benchmarks;

  if (!benchmarks[benchmark]) {
    benchmarks[benchmark] = [];
  }

  benchmarks[benchmark].push(rule.attributes.metadata.rego_rule_id);
  return benchmarks;
}, {});
/* @internal */


const createRulesConfig = cspRules => ({
  runtime_cfg: {
    activated_rules: getEnabledRulesByBenchmark(cspRules)
  }
});
/* @internal */


exports.createRulesConfig = createRulesConfig;

const getUpdatedPackagePolicy = (packagePolicy, runtimeCfg) => (0, _immer.default)(packagePolicy, draft => {
  (0, _lodash.unset)(draft, 'id');
  if (!draft.vars) draft.vars = {};
  draft.vars.runtimeCfg = {
    type: 'yaml',
    value: runtimeCfg
  };
});
/**
 * gets all rules of a package policy with fields required for runtime config
 */


exports.getUpdatedPackagePolicy = getUpdatedPackagePolicy;

const getCspRulesSO = (soClient, packagePolicyId, policyId) => soClient.find({
  type: _constants.CSP_RULE_SAVED_OBJECT_TYPE,
  filter: (0, _helpers.createCspRuleSearchFilterByPackagePolicy)({
    packagePolicyId,
    policyId
  }),
  fields: RUNTIME_CFG_FIELDS,
  perPage: 10000
}).then(response => response.saved_objects);
/**
 * gets rules of a package policy by id with fields required for updating SO object
 */


exports.getCspRulesSO = getCspRulesSO;

const getByIdCurrentRulesSO = (soClient, ruleIds) => soClient.bulkGet(ruleIds.map(ruleId => ({
  id: ruleId,
  type: _constants.CSP_RULE_SAVED_OBJECT_TYPE,
  fields: ['enabled'] // In case of a rollback, this is the only field we need to restore

}))).then(response => response.saved_objects);

const getRuntimeCfgVarValue = rules => _jsYaml.default.safeDump(createRulesConfig(rules));
/**
 * Updates the package policy vars object with a new value for the runtimeCfg key
 * @internal
 * */


const updatePackagePolicyRuntimeCfgVar = async ({
  packagePolicyService,
  packagePolicy,
  esClient,
  soClient,
  user,
  rules
}) => packagePolicyService.update(soClient, esClient, packagePolicy.id, getUpdatedPackagePolicy(packagePolicy, getRuntimeCfgVarValue(rules)), user ? {
  user
} : undefined);
/**
 * Keeps only fields we want/need to use for updating a rule saved-object
 */


exports.updatePackagePolicyRuntimeCfgVar = updatePackagePolicyRuntimeCfgVar;

const getRuleUpdateFields = ({
  id,
  type,
  attributes,
  references
}) => ({
  id,
  type,
  attributes,
  references
});
/**
 * Merges the current rules SO with their new 'enabled' value
 */


const getNextCspRulesSO = (current, rulesToChange) => {
  const currentRulesMap = Object.fromEntries(current.map(rule => [rule.id, getRuleUpdateFields(rule)]));
  return rulesToChange.map(rule => ({ ...currentRulesMap[rule.id],
    attributes: {
      enabled: rule.enabled
    }
  }));
};

const updateRulesSO = (soClient, rulesSO) => soClient.bulkUpdate(rulesSO);

const runUpdate = async ({
  rollbackSO,
  updateSO,
  updatePackagePolicy,
  logger
}) => {
  logger.info('Start updating rules');

  try {
    await updateSO();
  } catch (e) {
    logger.warn('Failed updating saved objects');
    logger.error(e);
    throw e;
  }

  try {
    const updatedPolicy = await updatePackagePolicy();
    logger.info('Finish updating rules');
    return updatedPolicy;
  } catch (e) {
    logger.warn('Failed updating package policy vars');
    logger.error(e);
    logger.info('Rollback to previous saved-objects rules');
    await rollbackSO();
    logger.info('Finish rollback');
    throw e;
  }
};
/**
 * Combines updating rules saved-objects and package policy vars into a single operation
 * @internal
 */


const updatePackagePolicyCspRules = async ({
  soClient,
  packagePolicyService,
  esClient,
  user,
  logger
}, packagePolicy, rulesToChange) => {
  // Get initial rules, before updating
  const currentRulesSO = await getByIdCurrentRulesSO(soClient, rulesToChange.map(rule => rule.id)); // Define actions to be executed for updating rules

  const updateSO = () => updateRulesSO(soClient, getNextCspRulesSO(currentRulesSO, rulesToChange));

  const updatePackagePolicy = async () => updatePackagePolicyRuntimeCfgVar({
    rules: await getCspRulesSO(soClient, packagePolicy.id, packagePolicy.policy_id),
    soClient,
    packagePolicyService,
    user,
    packagePolicy,
    esClient: esClient.asCurrentUser
  }); // Define actions to be executed for rollback


  const rollbackSO = () => updateRulesSO(soClient, currentRulesSO); // Start the update process


  return runUpdate({
    logger,
    updateSO,
    updatePackagePolicy,
    rollbackSO
  });
};
/** @internal */


exports.updatePackagePolicyCspRules = updatePackagePolicyCspRules;

const updateRulesConfigurationHandler = async (context, request, response) => {
  const cspContext = await context.csp;

  if (!(await context.fleet).authz.fleet.all) {
    return response.forbidden();
  }

  try {
    const packagePolicy = await cspContext.packagePolicyService.get(cspContext.soClient, request.body.package_policy_id);

    if (!packagePolicy) {
      // packagePolicyService.get() throws a 404 if the package policy is not found
      // we consider returning null as the same, and use the same error
      throw _server.SavedObjectsErrorHelpers.createGenericNotFoundError(_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE, request.body.package_policy_id);
    }

    cspContext.logger.info(`Start updating package policy - ${packagePolicy.id}`);
    const updatedPackagePolicy = await updatePackagePolicyCspRules(cspContext, packagePolicy, request.body.rules);
    return response.ok({
      body: updatedPackagePolicy
    });
  } catch (e) {
    const error = (0, _securitysolutionEsUtils.transformError)(e);
    cspContext.logger.error(`Failed updating package policy (${request.body.package_policy_id}) - ${error.message}`);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode
    });
  }
};

exports.updateRulesConfigurationHandler = updateRulesConfigurationHandler;

const defineUpdateRulesConfigRoute = router => router.post({
  path: _constants.UPDATE_RULES_CONFIG_ROUTE_PATH,
  validate: {
    body: updateRulesConfigurationBodySchema
  },
  options: {
    tags: ['access:cloud-security-posture-all']
  }
}, updateRulesConfigurationHandler);

exports.defineUpdateRulesConfigRoute = defineUpdateRulesConfigRoute;