"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createMlInferencePipeline = exports.createAndReferenceMlInferencePipeline = exports.addSubPipelineToIndexSpecificMlPipeline = void 0;

var _error_codes = require("../../common/types/error_codes");

var _create_pipeline_definitions = require("../lib/pipelines/create_pipeline_definitions");

var _ml_inference_pipeline_utils = require("./ml_inference_pipeline_utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates a Machine Learning Inference pipeline with the given settings, if it doesn't exist yet,
 * then references it in the "parent" ML Inference pipeline that is associated with the index.
 * @param indexName name of the index this pipeline corresponds to.
 * @param pipelineName pipeline name set by the user.
 * @param modelId model ID selected by the user.
 * @param sourceField The document field that model will read.
 * @param destinationField The document field that the model will write to.
 * @param esClient the Elasticsearch Client to use when retrieving pipeline and model details.
 */
const createAndReferenceMlInferencePipeline = async (indexName, pipelineName, modelId, sourceField, destinationField, esClient) => {
  const createPipelineResult = await createMlInferencePipeline(pipelineName, modelId, sourceField, destinationField, esClient);
  const addSubPipelineResult = await addSubPipelineToIndexSpecificMlPipeline(indexName, createPipelineResult.id, esClient);
  return Promise.resolve({ ...createPipelineResult,
    addedToParentPipeline: addSubPipelineResult.addedToParentPipeline
  });
};
/**
 * Creates a Machine Learning Inference pipeline with the given settings, if it doesn't exist yet.
 * @param pipelineName pipeline name set by the user.
 * @param modelId model ID selected by the user.
 * @param sourceField The document field that model will read.
 * @param destinationField The document field that the model will write to.
 * @param esClient the Elasticsearch Client to use when retrieving pipeline and model details.
 */


exports.createAndReferenceMlInferencePipeline = createAndReferenceMlInferencePipeline;

const createMlInferencePipeline = async (pipelineName, modelId, sourceField, destinationField, esClient) => {
  var _pipelineByName;

  const inferencePipelineGeneratedName = (0, _ml_inference_pipeline_utils.getPrefixedInferencePipelineProcessorName)(pipelineName); // Check that a pipeline with the same name doesn't already exist

  let pipelineByName;

  try {
    pipelineByName = await esClient.ingest.getPipeline({
      id: inferencePipelineGeneratedName
    });
  } catch (error) {// Silently swallow error
  }

  if ((_pipelineByName = pipelineByName) !== null && _pipelineByName !== void 0 && _pipelineByName[inferencePipelineGeneratedName]) {
    throw new Error(_error_codes.ErrorCode.PIPELINE_ALREADY_EXISTS);
  } // Generate pipeline with default processors


  const mlInferencePipeline = await (0, _create_pipeline_definitions.formatMlPipelineBody)(inferencePipelineGeneratedName, modelId, sourceField, destinationField || (0, _ml_inference_pipeline_utils.formatPipelineName)(pipelineName), esClient);
  await esClient.ingest.putPipeline({
    id: inferencePipelineGeneratedName,
    ...mlInferencePipeline
  });
  return Promise.resolve({
    id: inferencePipelineGeneratedName,
    created: true
  });
};
/**
 * Adds the supplied a Machine Learning Inference pipeline reference to the "parent" ML Inference
 * pipeline that is associated with the index.
 * @param indexName name of the index this pipeline corresponds to.
 * @param pipelineName name of the ML Inference pipeline to add.
 * @param esClient the Elasticsearch Client to use when retrieving pipeline details.
 */


exports.createMlInferencePipeline = createMlInferencePipeline;

const addSubPipelineToIndexSpecificMlPipeline = async (indexName, pipelineName, esClient) => {
  var _parentPipeline;

  const parentPipelineId = (0, _ml_inference_pipeline_utils.getInferencePipelineNameFromIndexName)(indexName); // Fetch the parent pipeline

  let parentPipeline;

  try {
    const pipelineResponse = await esClient.ingest.getPipeline({
      id: parentPipelineId
    });
    parentPipeline = pipelineResponse[parentPipelineId];
  } catch (error) {// Swallow error; in this case the next step will return
  } // Verify the parent pipeline exists with a processors array


  if (!((_parentPipeline = parentPipeline) !== null && _parentPipeline !== void 0 && _parentPipeline.processors)) {
    return Promise.resolve({
      id: pipelineName,
      addedToParentPipeline: false
    });
  } // Check if the sub-pipeline reference is already in the list of processors,
  // if so, don't modify it


  const existingSubPipeline = parentPipeline.processors.find(p => {
    var _p$pipeline;

    return ((_p$pipeline = p.pipeline) === null || _p$pipeline === void 0 ? void 0 : _p$pipeline.name) === pipelineName;
  });

  if (existingSubPipeline) {
    return Promise.resolve({
      id: pipelineName,
      addedToParentPipeline: false
    });
  } // Add sub-processor to the ML inference parent pipeline


  parentPipeline.processors.push({
    pipeline: {
      name: pipelineName
    }
  });
  await esClient.ingest.putPipeline({
    id: parentPipelineId,
    ...parentPipeline
  });
  return Promise.resolve({
    id: pipelineName,
    addedToParentPipeline: true
  });
};

exports.addSubPipelineToIndexSpecificMlPipeline = addSubPipelineToIndexSpecificMlPipeline;