"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDefaultSettings = createDefaultSettings;
exports.getCloudFleetServersHosts = getCloudFleetServersHosts;
exports.getSettings = getSettings;
exports.saveSettings = saveSettings;
exports.settingsSetup = settingsSetup;

var _boom = _interopRequireDefault(require("@hapi/boom"));

var _lodash = require("lodash");

var _services = require("../../common/services");

var _constants = require("../../common/constants");

var _app_context = require("./app_context");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
async function getSettings(soClient) {
  const res = await soClient.find({
    type: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
  });

  if (res.total === 0) {
    throw _boom.default.notFound('Global settings not found');
  }

  const settingsSo = res.saved_objects[0];
  return {
    id: settingsSo.id,
    ...settingsSo.attributes,
    fleet_server_hosts: settingsSo.attributes.fleet_server_hosts || [],
    preconfigured_fields: getConfigFleetServerHosts() ? ['fleet_server_hosts'] : []
  };
}

async function settingsSetup(soClient) {
  try {
    var _getConfigFleetServer, _getConfigFleetServer2;

    const settings = await getSettings(soClient);
    const defaultSettings = createDefaultSettings();
    const fleetServerHostsIsPreconfigured = (_getConfigFleetServer = (_getConfigFleetServer2 = getConfigFleetServerHosts()) === null || _getConfigFleetServer2 === void 0 ? void 0 : _getConfigFleetServer2.length) !== null && _getConfigFleetServer !== void 0 ? _getConfigFleetServer : 0 > 0;
    const fleetServerHostsShouldBeUpdated = !settings.fleet_server_hosts || settings.fleet_server_hosts.length === 0 || fleetServerHostsIsPreconfigured && !(0, _lodash.isEqual)(settings.fleet_server_hosts, defaultSettings.fleet_server_hosts); // Migration for < 7.13 Kibana

    if (defaultSettings.fleet_server_hosts.length > 0 && fleetServerHostsShouldBeUpdated) {
      return saveSettings(soClient, {
        fleet_server_hosts: defaultSettings.fleet_server_hosts
      });
    }
  } catch (e) {
    if (e.isBoom && e.output.statusCode === 404) {
      const defaultSettings = createDefaultSettings();
      return saveSettings(soClient, defaultSettings);
    }

    throw e;
  }
}

async function saveSettings(soClient, newData) {
  const data = { ...newData
  };

  if (data.fleet_server_hosts) {
    data.fleet_server_hosts = data.fleet_server_hosts.map(_services.normalizeHostsForAgents);
  }

  try {
    const settings = await getSettings(soClient);
    const res = await soClient.update(_constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE, settings.id, data);
    return {
      id: settings.id,
      ...res.attributes
    };
  } catch (e) {
    if (e.isBoom && e.output.statusCode === 404) {
      const defaultSettings = createDefaultSettings();
      const res = await soClient.create(_constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE, { ...defaultSettings,
        ...data
      }, {
        id: _constants.GLOBAL_SETTINGS_ID,
        overwrite: true
      });
      return {
        id: res.id,
        ...res.attributes
      };
    }

    throw e;
  }
}

function getConfigFleetServerHosts() {
  var _config$agents, _config$agents$fleet_, _config$agents2, _config$agents2$fleet;

  const config = _app_context.appContextService.getConfig();

  return config !== null && config !== void 0 && (_config$agents = config.agents) !== null && _config$agents !== void 0 && (_config$agents$fleet_ = _config$agents.fleet_server) !== null && _config$agents$fleet_ !== void 0 && _config$agents$fleet_.hosts && config.agents.fleet_server.hosts.length > 0 ? config === null || config === void 0 ? void 0 : (_config$agents2 = config.agents) === null || _config$agents2 === void 0 ? void 0 : (_config$agents2$fleet = _config$agents2.fleet_server) === null || _config$agents2$fleet === void 0 ? void 0 : _config$agents2$fleet.hosts : undefined;
}

function createDefaultSettings() {
  var _ref;

  const configFleetServerHosts = getConfigFleetServerHosts();
  const cloudFleetServerHosts = getCloudFleetServersHosts();
  const fleetServerHosts = (_ref = configFleetServerHosts !== null && configFleetServerHosts !== void 0 ? configFleetServerHosts : cloudFleetServerHosts) !== null && _ref !== void 0 ? _ref : [];
  return {
    fleet_server_hosts: fleetServerHosts
  };
}

function getCloudFleetServersHosts() {
  const cloudSetup = _app_context.appContextService.getCloud();

  if (cloudSetup && cloudSetup.isCloudEnabled && cloudSetup.cloudId && cloudSetup.deploymentId) {
    const res = (0, _services.decodeCloudId)(cloudSetup.cloudId);

    if (!res) {
      return;
    } // Fleet Server url are formed like this `https://<deploymentId>.fleet.<host>


    return [`https://${cloudSetup.deploymentId}.fleet.${res.host}${res.defaultPort !== '443' ? `:${res.defaultPort}` : ''}`];
  }
}