"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmTransactionErrorRateTransformGenerator = void 0;

var _slo_transform_template = require("../../../assets/transform_templates/slo_transform_template");

var _constants = require("../../../assets/constants");

var _models = require("../../../types/models");

var _schema = require("../../../types/schema");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const APM_SOURCE_INDEX = 'metrics-apm*';
const ALLOWED_STATUS_CODES = ['2xx', '3xx', '4xx', '5xx'];
const DEFAULT_GOOD_STATUS_CODES = ['2xx', '3xx', '4xx'];

class ApmTransactionErrorRateTransformGenerator {
  getTransformParams(slo, spaceId) {
    if (!_models.apmTransactionErrorRateSLOSchema.is(slo)) {
      throw new Error(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }

    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildSource(slo), this.buildDestination(slo, spaceId), this.buildGroupBy(), this.buildAggregations(slo));
  }

  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id);
  }

  buildSource(slo) {
    const queryFilter = [];

    if (slo.indicator.params.service !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.name': slo.indicator.params.service
        }
      });
    }

    if (slo.indicator.params.environment !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.environment': slo.indicator.params.environment
        }
      });
    }

    if (slo.indicator.params.transaction_name !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.name': slo.indicator.params.transaction_name
        }
      });
    }

    if (slo.indicator.params.transaction_type !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.type': slo.indicator.params.transaction_type
        }
      });
    }

    return {
      index: APM_SOURCE_INDEX,
      runtime_mappings: {
        'slo.id': {
          type: 'keyword',
          script: {
            source: `emit('${slo.id}')`
          }
        }
      },
      query: {
        bool: {
          filter: [{
            match: {
              'transaction.root': true
            }
          }, ...queryFilter]
        }
      }
    };
  }

  buildDestination(slo, spaceId) {
    return {
      pipeline: (0, _constants.getSLOIngestPipelineName)(spaceId),
      index: (0, _constants.getSLODestinationIndexName)(spaceId)
    };
  }

  buildGroupBy() {
    return {
      'slo.id': {
        terms: {
          field: 'slo.id'
        }
      },
      '@timestamp': {
        date_histogram: {
          field: '@timestamp',
          calendar_interval: '1m'
        }
      },
      'slo.context.transaction.name': {
        terms: {
          field: 'transaction.name'
        }
      },
      'slo.context.transaction.type': {
        terms: {
          field: 'transaction.type'
        }
      },
      'slo.context.service.name': {
        terms: {
          field: 'service.name'
        }
      },
      'slo.context.service.environment': {
        terms: {
          field: 'service.environment'
        }
      }
    };
  }

  buildAggregations(slo) {
    const goodStatusCodesFilter = this.getGoodStatusCodesFilter(slo.indicator.params.good_status_codes);
    return {
      'slo.numerator': {
        filter: {
          bool: {
            should: goodStatusCodesFilter
          }
        }
      },
      'slo.denominator': {
        value_count: {
          field: 'transaction.duration.histogram'
        }
      }
    };
  }

  getGoodStatusCodesFilter(goodStatusCodes) {
    let statusCodes = goodStatusCodes === null || goodStatusCodes === void 0 ? void 0 : goodStatusCodes.filter(code => ALLOWED_STATUS_CODES.includes(code));

    if (statusCodes === undefined || statusCodes.length === 0) {
      statusCodes = DEFAULT_GOOD_STATUS_CODES;
    }

    return statusCodes.map(code => ({
      match: {
        'transaction.result': `HTTP ${code}`
      }
    }));
  }

}

exports.ApmTransactionErrorRateTransformGenerator = ApmTransactionErrorRateTransformGenerator;