"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointActionGenerator = void 0;

var _lodash = require("lodash");

var _constants = require("../constants");

var _base_data_generator = require("./base_data_generator");

var _types = require("../types");

var _constants2 = require("../service/response_actions/constants");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
class EndpointActionGenerator extends _base_data_generator.BaseDataGenerator {
  /** Generate a random endpoint Action request (isolate or unisolate) */
  generate(overrides = {}) {
    const timeStamp = overrides['@timestamp'] ? new Date(overrides['@timestamp']) : new Date();
    return (0, _lodash.merge)({
      '@timestamp': timeStamp.toISOString(),
      agent: {
        id: [this.seededUUIDv4()]
      },
      EndpointActions: {
        action_id: this.seededUUIDv4(),
        expiration: this.randomFutureDate(timeStamp),
        type: 'INPUT_ACTION',
        input_type: 'endpoint',
        data: {
          command: this.randomResponseActionCommand(),
          comment: this.randomString(15),
          parameters: undefined
        }
      },
      error: undefined,
      user: {
        id: this.randomUser()
      }
    }, overrides);
  }

  generateActionEsHit(overrides = {}) {
    return Object.assign(this.toEsSearchHit(this.generate(overrides)), {
      _index: `.ds-${_constants.ENDPOINT_ACTIONS_DS}-some_namespace`
    });
  }
  /** Generates an endpoint action response */


  generateResponse(overrides = {}) {
    const timeStamp = overrides['@timestamp'] ? new Date(overrides['@timestamp']) : new Date();
    const startedAtTimes = [];
    [2, 3, 5, 8, 13, 21].forEach(n => {
      startedAtTimes.push(timeStamp.setMinutes(-this.randomN(n)), timeStamp.setSeconds(-this.randomN(n)));
    });
    return (0, _lodash.merge)({
      '@timestamp': timeStamp.toISOString(),
      agent: {
        id: this.seededUUIDv4()
      },
      EndpointActions: {
        action_id: this.seededUUIDv4(),
        completed_at: timeStamp.toISOString(),
        data: {
          command: this.randomResponseActionCommand(),
          comment: '',
          parameters: undefined
        },
        // randomly before a few hours/minutes/seconds later
        started_at: new Date(startedAtTimes[this.randomN(startedAtTimes.length)]).toISOString(),
        output: undefined
      },
      error: undefined
    }, overrides);
  }

  generateResponseEsHit(overrides = {}) {
    return Object.assign(this.toEsSearchHit(this.generateResponse(overrides)), {
      _index: `.ds-${_constants.ENDPOINT_ACTION_RESPONSES_DS}-some_namespace-something`
    });
  }

  generateActionDetails(overrides = {}) {
    const details = {
      agents: ['agent-a'],
      command: 'isolate',
      completedAt: '2022-04-30T16:08:47.449Z',
      hosts: {
        'agent-a': {
          name: 'Host-agent-a'
        }
      },
      id: '123',
      isCompleted: true,
      isExpired: false,
      wasSuccessful: true,
      errors: undefined,
      startedAt: '2022-04-27T16:08:47.449Z',
      status: 'successful',
      comment: 'thisisacomment',
      createdBy: 'auserid',
      parameters: undefined,
      outputs: {},
      agentState: {
        'agent-a': {
          errors: undefined,
          isCompleted: true,
          completedAt: '2022-04-30T16:08:47.449Z',
          wasSuccessful: true
        }
      }
    };
    return (0, _lodash.merge)(details, overrides);
  }

  generateActivityLogAction(overrides) {
    return (0, _lodash.merge)({
      type: _types.ActivityLogItemTypes.ACTION,
      item: {
        id: this.seededUUIDv4(),
        data: this.generate()
      }
    }, overrides);
  }

  generateActivityLogActionResponse(overrides) {
    return (0, _lodash.merge)({
      type: _types.ActivityLogItemTypes.RESPONSE,
      item: {
        id: this.seededUUIDv4(),
        data: this.generateResponse()
      }
    }, overrides);
  }

  generateAgentPendingActionsSummary(overrides = {}) {
    return (0, _lodash.merge)({
      agent_id: this.seededUUIDv4(),
      pending_actions: {
        isolate: 2,
        unisolate: 0
      }
    }, overrides);
  }

  randomFloat() {
    return this.random();
  }

  randomN(max) {
    return super.randomN(max);
  }

  randomResponseActionProcesses(n) {
    const numberOfEntries = n !== null && n !== void 0 ? n : this.randomChoice([20, 30, 40, 50]);
    const entries = [];

    for (let i = 0; i < numberOfEntries; i++) {
      entries.push({
        command: this.randomResponseActionProcessesCommand(),
        pid: this.randomN(1000).toString(),
        entity_id: this.randomString(50),
        user: this.randomUser()
      });
    }

    return entries;
  }

  randomResponseActionProcessesCommand() {
    const commands = ['/opt/cmd1', '/opt/cmd2', '/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3', '/opt/cmd3/opt/cmd3/opt/cmd3/opt/cmd3'];
    return this.randomChoice(commands);
  }

  randomResponseActionCommand() {
    return this.randomChoice(_constants2.RESPONSE_ACTION_COMMANDS);
  }

}

exports.EndpointActionGenerator = EndpointActionGenerator;