"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDefaultPolicy = void 0;

var _policy_config = require("../../../common/endpoint/models/policy_config");

var _license = require("../../../common/license/license");

var _types = require("../../../common/endpoint/types");

var _constants = require("../constants");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Create the default endpoint policy based on the current license and configuration type
 */
const createDefaultPolicy = (licenseService, config) => {
  const policy = (0, _license.isAtLeast)(licenseService.getLicenseInformation(), 'platinum') ? (0, _policy_config.policyFactory)() : (0, _policy_config.policyFactoryWithoutPaidFeatures)();

  if ((config === null || config === void 0 ? void 0 : config.type) === 'cloud') {
    return getCloudPolicyConfig(policy);
  }

  return getEndpointPolicyWithIntegrationConfig(policy, config);
};
/**
 * Set all keys of the given object to false
 */


exports.createDefaultPolicy = createDefaultPolicy;

const falsyObjectKeys = obj => {
  return Object.keys(obj).reduce((accumulator, key) => {
    return { ...accumulator,
      [key]: false
    };
  }, {});
};
/**
 * Retrieve policy for endpoint based on the preset selected in the endpoint integration config
 */


const getEndpointPolicyWithIntegrationConfig = (policy, config) => {
  var _config$endpointConfi, _config$endpointConfi2;

  const isEDREssential = (config === null || config === void 0 ? void 0 : (_config$endpointConfi = config.endpointConfig) === null || _config$endpointConfi === void 0 ? void 0 : _config$endpointConfi.preset) === _constants.ENDPOINT_CONFIG_PRESET_EDR_ESSENTIAL;

  if ((config === null || config === void 0 ? void 0 : (_config$endpointConfi2 = config.endpointConfig) === null || _config$endpointConfi2 === void 0 ? void 0 : _config$endpointConfi2.preset) === _constants.ENDPOINT_CONFIG_PRESET_NGAV || isEDREssential) {
    const events = {
      process: true,
      file: isEDREssential,
      network: isEDREssential
    };
    return { ...policy,
      linux: { ...policy.linux,
        events: { ...falsyObjectKeys(policy.linux.events),
          ...events
        }
      },
      windows: { ...policy.windows,
        events: { ...falsyObjectKeys(policy.windows.events),
          ...events
        }
      },
      mac: { ...policy.mac,
        events: { ...falsyObjectKeys(policy.mac.events),
          ...events
        }
      }
    };
  }

  return policy;
};
/**
 * Retrieve policy for cloud based on the on the cloud integration config
 */


const getCloudPolicyConfig = policy => {
  // Disabling all protections, since it's not yet supported on Cloud integrations
  const protections = {
    memory_protection: {
      supported: false,
      mode: _types.ProtectionModes.off
    },
    malware: { ...policy.linux.malware,
      mode: _types.ProtectionModes.off
    },
    behavior_protection: { ...policy.linux.behavior_protection,
      mode: _types.ProtectionModes.off
    }
  };
  return { ...policy,
    linux: { ...policy.linux,
      ...protections,
      events: { ...policy.linux.events,
        session_data: true
      }
    },
    windows: { ...policy.windows,
      ...protections,
      // Disabling ransomware protection, since it's not supported on Cloud integrations
      ransomware: {
        supported: false,
        mode: _types.ProtectionModes.off
      }
    },
    mac: { ...policy.mac,
      ...protections
    }
  };
};