"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.patchRulesRoute = void 0;

var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");

var _patch_rules_type_dependents = require("../../../../../common/detection_engine/schemas/request/patch_rules_type_dependents");

var _route_validation = require("../../../../utils/build_validation/route_validation");

var _patch_rules_schema = require("../../../../../common/detection_engine/schemas/request/patch_rules_schema");

var _constants = require("../../../../../common/constants");

var _authz = require("../../../machine_learning/authz");

var _validation = require("../../../machine_learning/validation");

var _patch_rules = require("../../rules/patch_rules");

var _utils = require("../utils");

var _check_for_default_rule_exception_list = require("./utils/check_for_default_rule_exception_list");

var _utils2 = require("./utils");

var _validate = require("./validate");

var _read_rules = require("../../rules/read_rules");

var _utils3 = require("../../rules/utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const patchRulesRoute = (router, ml) => {
  router.patch({
    path: _constants.DETECTION_ENGINE_RULES_URL,
    validate: {
      // Use non-exact validation because everything is optional in patch - since everything is optional,
      // io-ts can't find the right schema from the type specific union and the exact check breaks.
      // We do type specific validation after fetching the existing rule so we know the rule type.
      body: (0, _route_validation.buildRouteValidationNonExact)(_patch_rules_schema.patchRulesSchema)
    },
    options: {
      tags: ['access:securitySolution']
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const validationErrors = (0, _patch_rules_type_dependents.patchRuleValidateTypeDependents)(request.body);

    if (validationErrors.length) {
      return siemResponse.error({
        statusCode: 400,
        body: validationErrors
      });
    }

    try {
      const params = request.body;
      const rulesClient = (await context.alerting).getRulesClient();
      const ruleExecutionLog = (await context.securitySolution).getRuleExecutionLog();
      const savedObjectsClient = (await context.core).savedObjects.client;
      const mlAuthz = (0, _authz.buildMlAuthz)({
        license: (await context.licensing).license,
        ml,
        request,
        savedObjectsClient
      });

      if (params.type) {
        // reject an unauthorized "promotion" to ML
        (0, _validation.throwAuthzError)(await mlAuthz.validateRuleType(params.type));
      }

      const existingRule = await (0, _read_rules.readRules)({
        rulesClient,
        ruleId: params.rule_id,
        id: params.id
      });

      if (existingRule !== null && existingRule !== void 0 && existingRule.params.type) {
        // reject an unauthorized modification of an ML rule
        (0, _validation.throwAuthzError)(await mlAuthz.validateRuleType(existingRule === null || existingRule === void 0 ? void 0 : existingRule.params.type));
      }

      (0, _check_for_default_rule_exception_list.checkDefaultRuleExceptionListReferences)({
        exceptionLists: params.exceptions_list
      });
      const migratedRule = await (0, _utils3.legacyMigrate)({
        rulesClient,
        savedObjectsClient,
        rule: existingRule
      });
      const rule = await (0, _patch_rules.patchRules)({
        rulesClient,
        existingRule: migratedRule,
        nextParams: params
      });

      if (rule != null && rule.enabled != null && rule.name != null) {
        const ruleExecutionSummary = await ruleExecutionLog.getExecutionSummary(rule.id);
        const [validated, errors] = (0, _validate.transformValidate)(rule, ruleExecutionSummary);

        if (errors != null) {
          return siemResponse.error({
            statusCode: 500,
            body: errors
          });
        } else {
          return response.ok({
            body: validated !== null && validated !== void 0 ? validated : {}
          });
        }
      } else {
        const error = (0, _utils2.getIdError)({
          id: params.id,
          ruleId: params.rule_id
        });
        return siemResponse.error({
          body: error.message,
          statusCode: error.statusCode
        });
      }
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};

exports.patchRulesRoute = patchRulesRoute;