"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSyntheticsMonitorRoute = exports.getSyntheticsMonitorOverviewRoute = exports.getAllSyntheticsMonitorRoute = exports.findLocationItem = void 0;

var _configSchema = require("@kbn/config-schema");

var _server = require("../../../../../../src/core/server");

var _runtime_types = require("../../../common/runtime_types");

var _saved_objects = require("../../../common/types/saved_objects");

var _constants = require("../../../common/constants");

var _synthetics_monitor = require("../../legacy_uptime/lib/saved_objects/synthetics_monitor");

var _service_errors = require("../synthetics_service/service_errors");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const querySchema = _configSchema.schema.object({
  page: _configSchema.schema.maybe(_configSchema.schema.number()),
  perPage: _configSchema.schema.maybe(_configSchema.schema.number()),
  sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
  sortOrder: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('desc'), _configSchema.schema.literal('asc')])),
  query: _configSchema.schema.maybe(_configSchema.schema.string()),
  filter: _configSchema.schema.maybe(_configSchema.schema.string()),
  tags: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
  monitorType: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
  locations: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
  status: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())]))
});

const getMonitors = (request, syntheticsService, savedObjectsClient) => {
  const {
    perPage = 50,
    page,
    sortField,
    sortOrder,
    query,
    tags,
    monitorType,
    locations,
    filter = ''
  } = request;
  const locationFilter = parseLocationFilter(syntheticsService.locations, locations);
  const filters = getFilter('tags', tags) + getFilter('type', monitorType) + getFilter('locations.id', locationFilter);
  return savedObjectsClient.find({
    type: _synthetics_monitor.syntheticsMonitorType,
    perPage,
    page,
    sortField: sortField === 'schedule.keyword' ? 'schedule.number' : sortField,
    sortOrder,
    searchFields: ['name', 'tags.text', 'locations.id.text', 'urls'],
    search: query ? `${query}*` : undefined,
    filter: filters + filter
  });
};

const getSyntheticsMonitorRoute = libs => ({
  method: 'GET',
  path: _constants.API_URLS.SYNTHETICS_MONITORS + '/{monitorId}',
  validate: {
    params: _configSchema.schema.object({
      monitorId: _configSchema.schema.string({
        minLength: 1,
        maxLength: 1024
      })
    })
  },
  handler: async ({
    request,
    response,
    server: {
      encryptedSavedObjects
    },
    savedObjectsClient
  }) => {
    const {
      monitorId
    } = request.params;
    const encryptedSavedObjectsClient = encryptedSavedObjects.getClient();

    try {
      return await libs.requests.getSyntheticsMonitor({
        monitorId,
        encryptedSavedObjectsClient,
        savedObjectsClient
      });
    } catch (getErr) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
        return (0, _service_errors.getMonitorNotFoundResponse)(response, monitorId);
      }

      throw getErr;
    }
  }
});

exports.getSyntheticsMonitorRoute = getSyntheticsMonitorRoute;

const getAllSyntheticsMonitorRoute = () => ({
  method: 'GET',
  path: _constants.API_URLS.SYNTHETICS_MONITORS,
  validate: {
    query: querySchema
  },
  handler: async ({
    request,
    savedObjectsClient,
    syntheticsMonitorClient
  }) => {
    const {
      filters,
      query
    } = request.query;
    const monitorsPromise = getMonitors(request.query, syntheticsMonitorClient.syntheticsService, savedObjectsClient);

    if (filters || query) {
      const totalMonitorsPromise = savedObjectsClient.find({
        type: _synthetics_monitor.syntheticsMonitorType,
        perPage: 0,
        page: 1
      });
      const allResolved = await Promise.all([monitorsPromise, totalMonitorsPromise]);
      const {
        saved_objects: monitors,
        per_page: perPageT,
        ...rest
      } = allResolved[0];
      const {
        total
      } = allResolved[1];
      return { ...rest,
        monitors,
        perPage: perPageT,
        absoluteTotal: total,
        syncErrors: syntheticsMonitorClient.syntheticsService.syncErrors
      };
    }

    const {
      saved_objects: monitors,
      per_page: perPageT,
      ...rest
    } = await monitorsPromise;
    return { ...rest,
      monitors,
      perPage: perPageT,
      absoluteTotal: rest.total,
      syncErrors: syntheticsMonitorClient.syntheticsService.syncErrors
    };
  }
});

exports.getAllSyntheticsMonitorRoute = getAllSyntheticsMonitorRoute;

const getFilter = (field, values, operator = 'OR') => {
  if (!values) {
    return '';
  }

  const fieldKey = `${_saved_objects.monitorAttributes}.${field}`;

  if (Array.isArray(values)) {
    return `${fieldKey}:${values.join(` ${operator} ${fieldKey}:`)}`;
  }

  return `${fieldKey}:${values}`;
};

const parseLocationFilter = (serviceLocations, locations) => {
  var _findLocationItem$id2, _findLocationItem2;

  if (!locations) {
    return '';
  }

  if (Array.isArray(locations)) {
    return locations.map(loc => {
      var _findLocationItem$id, _findLocationItem;

      return (_findLocationItem$id = (_findLocationItem = findLocationItem(loc, serviceLocations)) === null || _findLocationItem === void 0 ? void 0 : _findLocationItem.id) !== null && _findLocationItem$id !== void 0 ? _findLocationItem$id : '';
    }).filter(val => !val);
  }

  return (_findLocationItem$id2 = (_findLocationItem2 = findLocationItem(locations, serviceLocations)) === null || _findLocationItem2 === void 0 ? void 0 : _findLocationItem2.id) !== null && _findLocationItem$id2 !== void 0 ? _findLocationItem$id2 : '';
};

const findLocationItem = (query, locations) => {
  return locations.find(({
    id,
    label
  }) => query === id || label === query);
};

exports.findLocationItem = findLocationItem;

const getSyntheticsMonitorOverviewRoute = () => ({
  method: 'GET',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_OVERVIEW,
  validate: {
    query: querySchema
  },
  handler: async ({
    request,
    savedObjectsClient,
    syntheticsMonitorClient
  }) => {
    const {
      perPage = 5
    } = request.query;
    const {
      saved_objects: monitors
    } = await getMonitors({
      perPage: 1000,
      sortField: 'name.keyword',
      sortOrder: 'asc',
      page: 1
    }, syntheticsMonitorClient.syntheticsService, savedObjectsClient);
    const allMonitorIds = [];
    const pages = {};
    let currentPage = 0;
    let currentItem = 0;
    let total = 0;
    monitors.forEach(monitor => {
      /* collect all monitor ids for use
       * in filtering overview requests */
      const id = monitor.id;
      allMonitorIds.push(id);
      /* for reach location, add a config item */

      const locations = monitor.attributes[_runtime_types.ConfigKey.LOCATIONS];
      locations.forEach(location => {
        const config = {
          id,
          name: monitor.attributes[_runtime_types.ConfigKey.NAME],
          location,
          isEnabled: monitor.attributes[_runtime_types.ConfigKey.ENABLED]
        };

        if (!pages[currentPage]) {
          pages[currentPage] = [config];
        } else {
          pages[currentPage].push(config);
        }

        currentItem++;
        total++;

        if (currentItem % perPage === 0) {
          currentPage++;
          currentItem = 0;
        }
      });
    });
    return {
      pages,
      total,
      allMonitorIds
    };
  }
});

exports.getSyntheticsMonitorOverviewRoute = getSyntheticsMonitorOverviewRoute;