import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _createClass from "@babel/runtime/helpers/createClass";
import _classCallCheck from "@babel/runtime/helpers/classCallCheck";
import _inherits from "@babel/runtime/helpers/inherits";
import _createSuper from "@babel/runtime/helpers/createSuper";
import _typeof from "@babel/runtime/helpers/typeof";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
import crypto from 'crypto';
import { Stream } from 'stream';
import * as zlib from 'zlib';
import { acceptCompression } from './accept_compression'; // type guard to identify compressed stream

function isCompressedSream(arg) {
  return _typeof(arg) === 'object' && arg !== null && typeof arg.flush === 'function';
}

var FLUSH_PAYLOAD_SIZE = 4 * 1024;

var UncompressedResponseStream = /*#__PURE__*/function (_Stream$PassThrough) {
  _inherits(UncompressedResponseStream, _Stream$PassThrough);

  var _super = _createSuper(UncompressedResponseStream);

  function UncompressedResponseStream() {
    _classCallCheck(this, UncompressedResponseStream);

    return _super.apply(this, arguments);
  }

  return _createClass(UncompressedResponseStream);
}(Stream.PassThrough);

var DELIMITER = '\n';

/**
 * Sets up a response stream with support for gzip compression depending on provided
 * request headers. Any non-string data pushed to the stream will be stream as NDJSON.
 *
 * @param headers - Request headers.
 * @param logger - Kibana logger.
 * @param compressOverride - Optional flag to override header based compression setting.
 * @param flushFix - Adds an attribute with a random string payload to overcome buffer flushing with certain proxy configurations.
 *
 * @returns An object with stream attributes and methods.
 */
export function streamFactory(headers, logger) {
  var compressOverride = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
  var flushFix = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;
  var streamType;
  var isCompressed = compressOverride && acceptCompression(headers);
  var stream = isCompressed ? zlib.createGzip() : new UncompressedResponseStream(); // If waiting for draining of the stream, items will be added to this buffer.

  var backPressureBuffer = []; // Flag will be set when the "drain" listener is active so we can avoid setting multiple listeners.

  var waitForDrain = false; // Instead of a flag this is an array where we check if we are waiting on any callback from writing to the stream.
  // It needs to be an array to avoid running into race conditions.

  var waitForCallbacks = []; // Flag to set if the stream should be ended. Because there could be items in the backpressure buffer, we might
  // not want to end the stream right away. Once the backpressure buffer is cleared, we'll end the stream eventually.

  var tryToEnd = false;

  function logDebugMessage(msg) {
    logger.debug("HTTP Response Stream: ".concat(msg));
  }

  function end() {
    tryToEnd = true;
    logDebugMessage("backPressureBuffer size on end(): ".concat(backPressureBuffer.length));
    logDebugMessage("waitForCallbacks size on end(): ".concat(waitForCallbacks.length)); // Before ending the stream, we need to empty the backPressureBuffer

    if (backPressureBuffer.length > 0) {
      var el = backPressureBuffer.shift();

      if (el !== undefined) {
        push(el, true);
      }

      return;
    }

    if (waitForCallbacks.length === 0) {
      logDebugMessage('All backPressureBuffer and waitForCallbacks cleared, ending the stream.');
      stream.end();
    }
  }

  function push(d) {
    var drain = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    logDebugMessage("Push to stream. Current backPressure buffer size: ".concat(backPressureBuffer.length, ", drain flag: ").concat(drain));

    if (d === undefined) {
      logger.error('Stream chunk must not be undefined.');
      return;
    } // Initialize the stream type with the first push to the stream,
    // otherwise check the integrity of the data to be pushed.


    if (streamType === undefined) {
      streamType = typeof d === 'string' ? 'string' : 'ndjson';
    } else if (streamType === 'string' && typeof d !== 'string') {
      logger.error('Must not push non-string chunks to a string based stream.');
      return;
    } else if (streamType === 'ndjson' && typeof d === 'string') {
      logger.error('Must not push raw string chunks to an NDJSON based stream.');
      return;
    }

    if (!drain && waitForDrain || !drain && backPressureBuffer.length > 0) {
      logDebugMessage('Adding item to backpressure buffer.');
      backPressureBuffer.push(d);
      return;
    }

    try {
      var line = streamType === 'ndjson' ? "".concat(JSON.stringify(_objectSpread(_objectSpread({}, d), flushFix ? {
        flushPayload: crypto.randomBytes(FLUSH_PAYLOAD_SIZE).toString('hex')
      } : {}))).concat(DELIMITER) : d;
      waitForCallbacks.push(1);
      var writeOk = stream.write(line, function () {
        waitForCallbacks.pop(); // Calling .flush() on a compression stream will
        // make zlib return as much output as currently possible.

        if (isCompressedSream(stream)) {
          stream.flush();
        }

        if (tryToEnd && waitForCallbacks.length === 0) {
          end();
        }
      });
      logDebugMessage("Ok to write to the stream again? ".concat(writeOk));

      if (!writeOk) {
        logDebugMessage("Should we add the \"drain\" listener?: ".concat(!waitForDrain));

        if (!waitForDrain) {
          waitForDrain = true;
          stream.once('drain', function () {
            logDebugMessage('The "drain" listener triggered, we can continue pushing to the stream.');
            waitForDrain = false;

            if (backPressureBuffer.length > 0) {
              var el = backPressureBuffer.shift();

              if (el !== undefined) {
                push(el, true);
              }
            }
          });
        }
      } else if (writeOk && drain && backPressureBuffer.length > 0) {
        logDebugMessage('Continue clearing the backpressure buffer.');
        var el = backPressureBuffer.shift();

        if (el !== undefined) {
          push(el, true);
        }
      }
    } catch (e) {
      logger.error("Could not serialize or stream data chunk: ".concat(e.toString()));
      return;
    }
  }

  var responseWithHeaders = {
    body: stream,
    headers: _objectSpread(_objectSpread({}, isCompressed ? {
      'content-encoding': 'gzip'
    } : {}), {}, {
      // This disables response buffering on proxy servers (Nginx, uwsgi, fastcgi, etc.)
      // Otherwise, those proxies buffer responses up to 4/8 KiB.
      'X-Accel-Buffering': 'no',
      'Cache-Control': 'no-cache',
      Connection: 'keep-alive',
      'Transfer-Encoding': 'chunked'
    })
  };
  return {
    DELIMITER: DELIMITER,
    end: end,
    push: push,
    responseWithHeaders: responseWithHeaders
  };
}