import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _classCallCheck from "@babel/runtime/helpers/classCallCheck";
import _createClass from "@babel/runtime/helpers/createClass";
import _defineProperty from "@babel/runtime/helpers/defineProperty";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { BehaviorSubject, Subject, combineLatest, from, merge } from 'rxjs';
import { buffer, bufferCount, concatMap, delay, filter, groupBy, map, mergeMap, share, shareReplay, skipWhile, takeUntil, tap } from 'rxjs/operators';
import { ShippersRegistry } from './shippers_registry';
import { OptInConfigService } from './opt_in_config';
import { ContextService } from './context_service';
import { schemaToIoTs, validateSchema } from '../schema/validation';
export var AnalyticsClient = /*#__PURE__*/function () {
  // Using `share` so we can have multiple subscribers

  /**
   * This queue holds all the events until both conditions occur:
   * 1. We know the user's optIn decision.
   * 2. We have, at least, one registered shipper.
   * @private
   */

  /**
   * Observable used to report when a shipper is registered.
   * @private
   */
  function AnalyticsClient(initContext) {
    var _this = this;

    _classCallCheck(this, AnalyticsClient);

    _defineProperty(this, "internalTelemetryCounter$", new Subject());

    _defineProperty(this, "telemetryCounter$", this.internalTelemetryCounter$.pipe(share()));

    _defineProperty(this, "internalEventQueue$", new Subject());

    _defineProperty(this, "shippersRegistry", new ShippersRegistry());

    _defineProperty(this, "shipperRegistered$", new Subject());

    _defineProperty(this, "eventTypeRegistry", new Map());

    _defineProperty(this, "contextService", void 0);

    _defineProperty(this, "context$", new BehaviorSubject({}));

    _defineProperty(this, "optInConfig$", new BehaviorSubject(undefined));

    _defineProperty(this, "optInConfigWithReplay$", this.optInConfig$.pipe(filter(function (optInConfig) {
      return typeof optInConfig !== 'undefined';
    }), shareReplay(1)));

    _defineProperty(this, "contextWithReplay$", this.context$.pipe(skipWhile(function () {
      return !_this.optInConfig$.value;
    }), // Do not forward the context events until we have an optInConfig value
    shareReplay(1)));

    _defineProperty(this, "reportEvent", function (eventType, eventData) {
      // Fetch the timestamp as soon as we receive the event.
      var timestamp = new Date().toISOString();

      _this.internalTelemetryCounter$.next({
        type: 'enqueued',
        source: 'client',
        event_type: eventType,
        code: 'enqueued',
        count: 1
      });

      var eventTypeOpts = _this.eventTypeRegistry.get(eventType);

      if (!eventTypeOpts) {
        _this.internalTelemetryCounter$.next({
          type: 'dropped',
          source: 'client',
          event_type: eventType,
          code: 'UnregisteredType',
          count: 1
        });

        throw new Error("Attempted to report event type \"".concat(eventType, "\", before registering it. Use the \"registerEventType\" API to register it."));
      } // If the validator is registered (dev-mode only), perform the validation.


      if (eventTypeOpts.validator) {
        validateSchema("Event Type '".concat(eventType, "'"), eventTypeOpts.validator, eventData);
      }

      var event = {
        timestamp: timestamp,
        event_type: eventType,
        context: _this.context$.value,
        properties: eventData
      }; // debug-logging before checking the opt-in status to help during development

      if (_this.initContext.isDev) {
        _this.initContext.logger.debug("Report event \"".concat(eventType, "\""), {
          ebt_event: event
        });
      }

      var optInConfig = _this.optInConfig$.value;

      if ((optInConfig === null || optInConfig === void 0 ? void 0 : optInConfig.isEventTypeOptedIn(eventType)) === false) {
        // If opted out, skip early
        return;
      }

      if (typeof optInConfig === 'undefined') {
        // If the opt-in config is not provided yet, we need to enqueue the event to an internal queue
        _this.internalEventQueue$.next(event);
      } else {
        _this.sendToShipper(eventType, [event]);
      }
    });

    _defineProperty(this, "registerEventType", function (eventTypeOps) {
      if (_this.eventTypeRegistry.get(eventTypeOps.eventType)) {
        throw new Error("Event Type \"".concat(eventTypeOps.eventType, "\" is already registered."));
      }

      _this.eventTypeRegistry.set(eventTypeOps.eventType, _objectSpread(_objectSpread({}, eventTypeOps), {}, {
        validator: _this.initContext.isDev ? schemaToIoTs(eventTypeOps.schema) : undefined
      }));
    });

    _defineProperty(this, "optIn", function (optInConfig) {
      var optInConfigInstance = new OptInConfigService(optInConfig);

      _this.optInConfig$.next(optInConfigInstance);
    });

    _defineProperty(this, "registerContextProvider", function (contextProviderOpts) {
      _this.contextService.registerContextProvider(contextProviderOpts);
    });

    _defineProperty(this, "removeContextProvider", function (name) {
      _this.contextService.removeContextProvider(name);
    });

    _defineProperty(this, "registerShipper", function (ShipperClass, shipperConfig) {
      var _shipper$telemetryCou;

      var _ref = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {},
          _ref$exclusiveEventTy = _ref.exclusiveEventTypes,
          exclusiveEventTypes = _ref$exclusiveEventTy === void 0 ? [] : _ref$exclusiveEventTy;

      var shipperName = ShipperClass.shipperName;
      var shipper = new ShipperClass(shipperConfig, _objectSpread(_objectSpread({}, _this.initContext), {}, {
        logger: _this.initContext.logger.get('shipper', shipperName)
      }));

      if (exclusiveEventTypes.length) {
        // This feature is not intended to be supported in the MVP.
        // I can remove it if we think it causes more bad than good.
        exclusiveEventTypes.forEach(function (eventType) {
          _this.shippersRegistry.addEventExclusiveShipper(eventType, shipperName, shipper);
        });
      } else {
        _this.shippersRegistry.addGlobalShipper(shipperName, shipper);
      } // Subscribe to the shipper's telemetryCounter$ and pass it over to the client's-level observable


      (_shipper$telemetryCou = shipper.telemetryCounter$) === null || _shipper$telemetryCou === void 0 ? void 0 : _shipper$telemetryCou.subscribe(function (counter) {
        return _this.internalTelemetryCounter$.next(_objectSpread(_objectSpread({}, counter), {}, {
          source: shipperName // Enforce the shipper's name in the `source`

        }));
      }); // Spread the optIn configuration updates

      _this.optInConfigWithReplay$.subscribe(function (optInConfig) {
        var isOptedIn = optInConfig.isShipperOptedIn(shipperName);

        try {
          shipper.optIn(isOptedIn);
        } catch (err) {
          _this.initContext.logger.warn("Failed to set isOptedIn:".concat(isOptedIn, " in shipper ").concat(shipperName), err);
        }
      }); // Spread the global context if it has custom extendContext method


      if (shipper.extendContext) {
        _this.contextWithReplay$.subscribe(function (context) {
          try {
            shipper.extendContext(context);
          } catch (err) {
            _this.initContext.logger.warn("Shipper \"".concat(shipperName, "\" failed to extend the context"), err);
          }
        });
      } // Notify that a shipper is registered


      _this.shipperRegistered$.next();
    });

    _defineProperty(this, "shutdown", function () {
      _this.shippersRegistry.allShippers.forEach(function (shipper, shipperName) {
        try {
          shipper.shutdown();
        } catch (err) {
          _this.initContext.logger.warn("Failed to shutdown shipper \"".concat(shipperName, "\""), err);
        }
      });

      _this.internalEventQueue$.complete();

      _this.internalTelemetryCounter$.complete();

      _this.shipperRegistered$.complete();

      _this.optInConfig$.complete();

      _this.context$.complete();
    });

    this.initContext = initContext;
    this.contextService = new ContextService(this.context$, this.initContext.isDev, this.initContext.logger.get('context-service'));
    this.reportEnqueuedEventsWhenClientIsReady();
  }

  _createClass(AnalyticsClient, [{
    key: "sendToShipper",
    value:
    /**
     * Forwards the `events` to the registered shippers, bearing in mind if the shipper is opted-in for that eventType.
     * @param eventType The event type's name
     * @param events A bulk array of events matching the eventType.
     * @private
     */
    function sendToShipper(eventType, events) {
      var _this2 = this;

      var sentToShipper = false;
      this.shippersRegistry.getShippersForEventType(eventType).forEach(function (shipper, shipperName) {
        var _this2$optInConfig$$v;

        var isShipperOptedIn = (_this2$optInConfig$$v = _this2.optInConfig$.value) === null || _this2$optInConfig$$v === void 0 ? void 0 : _this2$optInConfig$$v.isShipperOptedIn(shipperName, eventType); // Only send it to the non-explicitly opted-out shippers

        if (isShipperOptedIn) {
          sentToShipper = true;

          try {
            shipper.reportEvents(events);
          } catch (err) {
            _this2.initContext.logger.warn("Failed to report event \"".concat(eventType, "\" via shipper \"").concat(shipperName, "\""), err);
          }
        }
      });

      if (sentToShipper) {
        this.internalTelemetryCounter$.next({
          type: 'sent_to_shipper',
          source: 'client',
          event_type: eventType,
          code: 'OK',
          count: events.length
        });
      }
    }
    /**
     * Once the client is ready (it has a valid optInConfig and at least one shipper),
     * flush any early events and ship them or discard them based on the optInConfig.
     * @private
     */

  }, {
    key: "reportEnqueuedEventsWhenClientIsReady",
    value: function reportEnqueuedEventsWhenClientIsReady() {
      var _this3 = this;

      // Observer that will emit when both events occur: the OptInConfig is set + a shipper has been registered
      var configReceivedAndShipperReceivedObserver$ = combineLatest([this.optInConfigWithReplay$, merge([this.shipperRegistered$, // Merging shipperRegistered$ with the optInConfigWithReplay$ when optedIn is false, so that we don't need to wait for the shipper if opted-in === false
      this.optInConfigWithReplay$.pipe(filter(function (cfg) {
        return (cfg === null || cfg === void 0 ? void 0 : cfg.isOptedIn()) === false;
      }))])]); // Flush the internal queue when we get any optInConfig and, at least, 1 shipper

      this.internalEventQueue$.pipe( // Take until will close the observer once we reach the condition below
      takeUntil(configReceivedAndShipperReceivedObserver$), // Accumulate the events until we can send them
      buffer(configReceivedAndShipperReceivedObserver$), // Minimal delay only to make this chain async and let the optIn operation to complete first.
      delay(0), // Re-emit the context to make sure all the shippers got it (only if opted-in)
      tap(function () {
        var _this3$optInConfig$$v;

        if ((_this3$optInConfig$$v = _this3.optInConfig$.value) !== null && _this3$optInConfig$$v !== void 0 && _this3$optInConfig$$v.isOptedIn()) {
          _this3.context$.next(_this3.context$.value);
        }
      }), // Minimal delay only to make this chain async and let
      // the context update operation to complete first.
      delay(0), // Flatten the array of events
      concatMap(function (events) {
        return from(events);
      }), // Discard opted-out events
      filter(function (event) {
        var _this3$optInConfig$$v2;

        return ((_this3$optInConfig$$v2 = _this3.optInConfig$.value) === null || _this3$optInConfig$$v2 === void 0 ? void 0 : _this3$optInConfig$$v2.isEventTypeOptedIn(event.event_type)) === true;
      }), // Let's group the requests per eventType for easier batching
      groupBy(function (event) {
        return event.event_type;
      }), mergeMap(function (groupedObservable) {
        return groupedObservable.pipe(bufferCount(1000), // Batching up-to 1000 events per event type for backpressure reasons
        map(function (events) {
          return {
            eventType: groupedObservable.key,
            events: events
          };
        }));
      })).subscribe(function (_ref2) {
        var eventType = _ref2.eventType,
            events = _ref2.events;

        _this3.sendToShipper(eventType, events);
      });
    }
  }]);

  return AnalyticsClient;
}();