"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaResponse = void 0;
exports.isKibanaResponse = isKibanaResponse;
exports.lifecycleResponseFactory = exports.kibanaResponseFactory = void 0;

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
function isKibanaResponse(response) {
  return typeof response.status === 'number' && typeof response.options === 'object';
}
/**
 * A response data object, expected to returned as a result of {@link RequestHandler} execution
 * @internal
 */


class KibanaResponse {
  constructor(status, payload, options = {}) {
    this.status = status;
    this.payload = payload;
    this.options = options;
  }

}

exports.KibanaResponse = KibanaResponse;
const successResponseFactory = {
  ok: (options = {}) => new KibanaResponse(200, options.body, options),
  accepted: (options = {}) => new KibanaResponse(202, options.body, options),
  noContent: (options = {}) => new KibanaResponse(204, undefined, options)
};
const redirectionResponseFactory = {
  redirected: options => new KibanaResponse(302, options.body, options)
};
const errorResponseFactory = {
  badRequest: (options = {}) => new KibanaResponse(400, options.body || 'Bad Request', options),
  unauthorized: (options = {}) => new KibanaResponse(401, options.body || 'Unauthorized', options),
  forbidden: (options = {}) => new KibanaResponse(403, options.body || 'Forbidden', options),
  notFound: (options = {}) => new KibanaResponse(404, options.body || 'Not Found', options),
  conflict: (options = {}) => new KibanaResponse(409, options.body || 'Conflict', options),
  customError: options => {
    if (!options || !options.statusCode) {
      throw new Error(`options.statusCode is expected to be set. given options: ${options && options.statusCode}`);
    }

    if (options.statusCode < 400 || options.statusCode >= 600) {
      throw new Error(`Unexpected Http status code. Expected from 400 to 599, but given: ${options.statusCode}`);
    }

    return new KibanaResponse(options.statusCode, options.body, options);
  }
};
const kibanaResponseFactory = { ...successResponseFactory,
  ...redirectionResponseFactory,
  ...errorResponseFactory,
  custom: options => {
    if (!options || !options.statusCode) {
      throw new Error(`options.statusCode is expected to be set. given options: ${options && options.statusCode}`);
    }

    const {
      statusCode: code,
      body,
      ...rest
    } = options;
    return new KibanaResponse(code, body, { ...rest
    });
  }
};
exports.kibanaResponseFactory = kibanaResponseFactory;
const lifecycleResponseFactory = { ...redirectionResponseFactory,
  ...errorResponseFactory
};
exports.lifecycleResponseFactory = lifecycleResponseFactory;