/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { deepFreeze } from '@kbn/std';
/**
 * The current status of a service at a point in time.
 *
 * @typeParam Meta - JSON-serializable object. Plugins should export this type to allow other plugins to read the `meta`
 *                   field in a type-safe way.
 * @public
 */

/**
 * The current "level" of availability of a service.
 *
 * @remarks
 * The values implement `valueOf` to allow for easy comparisons between status levels with <, >, etc. Higher values
 * represent higher severities. Note that the default `Array.prototype.sort` implementation does not correctly sort
 * these values.
 *
 * A snapshot serializer is available in `src/core/server/test_utils` to ease testing of these values with Jest.
 *
 * @public
 */
export var ServiceStatusLevels = deepFreeze({
  /**
   * Everything is working!
   */
  available: {
    toString: function toString() {
      return 'available';
    },
    valueOf: function valueOf() {
      return 0;
    },
    toJSON: function toJSON() {
      return this.toString();
    }
  },

  /**
   * Some features may not be working.
   */
  degraded: {
    toString: function toString() {
      return 'degraded';
    },
    valueOf: function valueOf() {
      return 1;
    },
    toJSON: function toJSON() {
      return this.toString();
    }
  },

  /**
   * The service is unavailable, but other functions that do not depend on this service should work.
   */
  unavailable: {
    toString: function toString() {
      return 'unavailable';
    },
    valueOf: function valueOf() {
      return 2;
    },
    toJSON: function toJSON() {
      return this.toString();
    }
  },

  /**
   * Block all user functions and display the status page, reserved for Core services only.
   */
  critical: {
    toString: function toString() {
      return 'critical';
    },
    valueOf: function valueOf() {
      return 3;
    },
    toJSON: function toJSON() {
      return this.toString();
    }
  }
});
/**
 * A convenience type that represents the union of each value in {@link ServiceStatusLevels}.
 * @public
 */