import _typeof from "@babel/runtime/helpers/typeof";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { getType } from '../get_type';
import { isAst } from './ast';
import { patch } from './patch';

function getArgumentString(arg, argKey) {
  var level = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
  var type = getType(arg); // eslint-disable-next-line @typescript-eslint/no-shadow

  function maybeArgKey(argKey, argString) {
    return argKey == null || argKey === '_' ? argString : "".concat(argKey, "=").concat(argString);
  }

  if (type === 'string') {
    // correctly (re)escape double quotes
    var escapedArg = arg.replace(/[\\"]/g, '\\$&'); // $& means the whole matched string

    return maybeArgKey(argKey, "\"".concat(escapedArg, "\""));
  }

  if (type === 'boolean' || type === 'null' || type === 'number') {
    // use values directly
    return maybeArgKey(argKey, "".concat(arg));
  }

  if (type === 'expression') {
    // build subexpressions
    return maybeArgKey(argKey, "{".concat(getExpression(arg.chain, level + 1), "}"));
  } // unknown type, throw with type value


  throw new Error("Invalid argument type in AST: ".concat(type));
}

function getExpressionArgs(_ref) {
  var args = _ref.arguments;
  var level = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;

  if (args == null || _typeof(args) !== 'object' || Array.isArray(args)) {
    throw new Error('Arguments can only be an object');
  }

  var argKeys = Object.keys(args);
  var MAX_LINE_LENGTH = 80; // length before wrapping arguments

  return argKeys.map(function (argKey) {
    return args[argKey].reduce(function (acc, arg) {
      var argString = getArgumentString(arg, argKey, level);
      var lineLength = acc.split('\n').pop().length; // if arg values are too long, move it to the next line

      if (level === 0 && lineLength + argString.length > MAX_LINE_LENGTH) {
        return "".concat(acc, "\n  ").concat(argString);
      } // append arg values to existing arg values


      if (lineLength > 0) {
        return "".concat(acc, " ").concat(argString);
      } // start the accumulator with the first arg value


      return argString;
    }, '');
  });
}

function fnWithArgs(fnName, args) {
  var _args$join;

  return "".concat(fnName, " ").concat((_args$join = args === null || args === void 0 ? void 0 : args.join(' ')) !== null && _args$join !== void 0 ? _args$join : '').trim();
}

function getExpression(chain) {
  var level = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;

  if (!chain) {
    throw new Error('Expressions must contain a chain');
  } // break new functions onto new lines if we're not in a nested/sub-expression


  var separator = level > 0 ? ' | ' : '\n| ';
  return chain.map(function (item) {
    var type = getType(item);

    if (type !== 'function') {
      return;
    }

    var fn = item["function"];

    if (!fn) {
      throw new Error('Functions must have a function name');
    }

    var expressionArgs = getExpressionArgs(item, level);
    return fnWithArgs(fn, expressionArgs);
  }).join(separator);
}

export function toExpression(ast) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'expression';

  var _ref2 = typeof options === 'string' ? {
    type: options
  } : options,
      type = _ref2.type,
      source = _ref2.source;

  if (source && isAst(ast)) {
    return patch(source, ast);
  }

  if (type === 'argument') {
    return getArgumentString(ast);
  }

  var nodeType = getType(ast);

  if (nodeType === 'expression') {
    var _ref3 = ast,
        chain = _ref3.chain;

    if (!Array.isArray(chain)) {
      throw new Error('Expressions must contain a chain');
    }

    return getExpression(chain);
  }

  if (nodeType === 'function') {
    var _ref4 = ast,
        fn = _ref4["function"];
    var args = getExpressionArgs(ast);
    return fnWithArgs(fn, args);
  }

  throw new Error('Expression must be an expression or argument function');
}