"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getChangePointRequest = exports.fetchChangePointPValues = void 0;

var _lodash = require("lodash");

var _constants = require("../../../common/constants");

var _get_query_with_params = require("./get_query_with_params");

var _get_request_base = require("./get_request_base");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
// TODO Consolidate with duplicate `fetchDurationFieldCandidates` in
// `x-pack/plugins/apm/server/routes/correlations/queries/fetch_failed_events_correlation_p_values.ts`
const getChangePointRequest = (params, fieldName) => {
  var _params$timeFieldName;

  const query = (0, _get_query_with_params.getQueryWithParams)({
    params
  });
  const timeFieldName = (_params$timeFieldName = params.timeFieldName) !== null && _params$timeFieldName !== void 0 ? _params$timeFieldName : '@timestamp';
  let filter = [];

  if (Array.isArray(query.bool.filter)) {
    filter = query.bool.filter.filter(d => Object.keys(d)[0] !== 'range');
    query.bool.filter = [...filter, {
      range: {
        [timeFieldName]: {
          gte: params.deviationMin,
          lt: params.deviationMax,
          format: 'epoch_millis'
        }
      }
    }];
  }

  const body = {
    query,
    size: 0,
    aggs: {
      change_point_p_value: {
        significant_terms: {
          field: fieldName,
          background_filter: {
            bool: {
              filter: [...filter, {
                range: {
                  [timeFieldName]: {
                    gte: params.baselineMin,
                    lt: params.baselineMax,
                    format: 'epoch_millis'
                  }
                }
              }]
            }
          },
          p_value: {
            background_is_superset: false
          },
          size: 1000
        }
      }
    }
  };
  return { ...(0, _get_request_base.getRequestBase)(params),
    body
  };
};

exports.getChangePointRequest = getChangePointRequest;

const fetchChangePointPValues = async (esClient, params, fieldNames, logger, emitError) => {
  const result = [];

  for (const fieldName of fieldNames) {
    const request = getChangePointRequest(params, fieldName);
    const resp = await esClient.search(request);

    if (resp.aggregations === undefined) {
      logger.error(`Failed to fetch p-value aggregation for fieldName "${fieldName}", got: \n${JSON.stringify(resp, null, 2)}`);
      emitError(`Failed to fetch p-value aggregation for fieldName "${fieldName}".`); // Still continue the analysis even if individual p-value queries fail.

      continue;
    }

    const overallResult = resp.aggregations.change_point_p_value;

    for (const bucket of overallResult.buckets) {
      const pValue = Math.exp(-bucket.score); // Scale the score into a value from 0 - 1
      // using a concave piecewise linear function in -log(p-value)

      const normalizedScore = 0.5 * Math.min(Math.max((bucket.score - 3.912) / 2.995, 0), 1) + 0.25 * Math.min(Math.max((bucket.score - 6.908) / 6.908, 0), 1) + 0.25 * Math.min(Math.max((bucket.score - 13.816) / 101.314, 0), 1);

      if (typeof pValue === 'number' && pValue < _constants.SPIKE_ANALYSIS_THRESHOLD) {
        result.push({
          fieldName,
          fieldValue: String(bucket.key),
          doc_count: bucket.doc_count,
          bg_count: bucket.bg_count,
          total_doc_count: overallResult.doc_count,
          total_bg_count: overallResult.bg_count,
          score: bucket.score,
          pValue,
          normalizedScore
        });
      }
    }
  }

  return (0, _lodash.uniqBy)(result, d => `${d.fieldName},${d.fieldValue}`);
};

exports.fetchChangePointPValues = fetchChangePointPValues;