"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleActionsForAlerts = scheduleActionsForAlerts;

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
async function scheduleActionsForAlerts(params) {
  const {
    logger,
    activeAlerts,
    recoveryActionGroup,
    recoveredAlerts,
    executionHandler,
    mutedAlertIdsSet,
    ruleLabel,
    ruleRunMetricsStore,
    throttle,
    notifyWhen
  } = params; // execute alerts with executable actions

  for (const [alertId, alert] of Object.entries(activeAlerts)) {
    const executeAction = shouldExecuteAction(alertId, alert, mutedAlertIdsSet, ruleLabel, logger, throttle, notifyWhen);

    if (executeAction && alert.hasScheduledActions()) {
      const {
        actionGroup,
        subgroup: actionSubgroup,
        state
      } = alert.getScheduledActionOptions();
      await executeAlert(alertId, alert, executionHandler, ruleRunMetricsStore, actionGroup, state, actionSubgroup);
    }
  } // execute recovered alerts


  for (const alertId of Object.keys(recoveredAlerts)) {
    if (mutedAlertIdsSet.has(alertId)) {
      logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${ruleLabel}: instance is muted`);
    } else {
      const alert = recoveredAlerts[alertId];
      await executeAlert(alertId, alert, executionHandler, ruleRunMetricsStore, recoveryActionGroup.id, {});
      alert.scheduleActions(recoveryActionGroup.id);
    }
  }
}

async function executeAlert(alertId, alert, executionHandler, ruleRunMetricsStore, actionGroup, state, actionSubgroup) {
  alert.updateLastScheduledActions(actionGroup, actionSubgroup);
  alert.unscheduleActions();
  return executionHandler({
    actionGroup,
    actionSubgroup,
    context: alert.getContext(),
    state,
    alertId,
    ruleRunMetricsStore
  });
}

function shouldExecuteAction(alertId, alert, mutedAlertIdsSet, ruleLabel, logger, throttle, notifyWhen) {
  const throttled = alert.isThrottled(throttle);
  const muted = mutedAlertIdsSet.has(alertId);
  let executeAction = true;

  if (throttled || muted) {
    executeAction = false;
    logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${ruleLabel}: rule is ${muted ? 'muted' : 'throttled'}`);
  } else if (notifyWhen === 'onActionGroupChange' && !alert.scheduledActionGroupOrSubgroupHasChanged()) {
    executeAction = false;
    logger.debug(`skipping scheduling of actions for '${alertId}' in rule ${ruleLabel}: alert is active but action group has not changed`);
  }

  return executeAction;
}