"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getComputeUsage = getComputeUsage;

var _i18n = require("@kbn/i18n");

var _common = require("../../../../../../observability/common");

var _server = require("../../../../../../observability/server");

var _uiTheme = require("@kbn/ui-theme");

var _elasticsearch_fieldnames = require("../../../../../common/elasticsearch_fieldnames");

var _environment_query = require("../../../../../common/utils/environment_query");

var _is_finite_number = require("../../../../../common/utils/is_finite_number");

var _metrics = require("../../../../lib/helpers/metrics");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * To calculate the compute usage we need to multiply the "system.memory.total" by "faas.billed_duration".
 * But the result of this calculation is in Bytes-milliseconds, as the "system.memory.total" is stored in bytes and the "faas.billed_duration" is stored in milliseconds.
 * But to calculate the overall cost AWS uses GB-second, so we need to convert the result to this unit.
 */
const GB = 1024 ** 3;

function calculateComputeUsageGBSeconds({
  faasBilledDuration,
  totalMemory
}) {
  if (!(0, _is_finite_number.isFiniteNumber)(faasBilledDuration) || !(0, _is_finite_number.isFiniteNumber)(totalMemory)) {
    return 0;
  }

  const totalMemoryGB = totalMemory / GB;
  const faasBilledDurationSec = faasBilledDuration / 1000;
  return totalMemoryGB * faasBilledDurationSec;
}

async function getComputeUsage({
  environment,
  kuery,
  setup,
  serviceName,
  start,
  end
}) {
  const {
    apmEventClient,
    config
  } = setup;
  const aggs = {
    avgFaasBilledDuration: {
      avg: {
        field: _elasticsearch_fieldnames.FAAS_BILLED_DURATION
      }
    },
    avgTotalMemory: {
      avg: {
        field: _elasticsearch_fieldnames.METRIC_SYSTEM_TOTAL_MEMORY
      }
    }
  };
  const params = {
    apm: {
      events: [_common.ProcessorEvent.metric]
    },
    body: {
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [{
            term: {
              [_elasticsearch_fieldnames.SERVICE_NAME]: serviceName
            }
          }, ...(0, _server.rangeQuery)(start, end), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), {
            exists: {
              field: _elasticsearch_fieldnames.FAAS_BILLED_DURATION
            }
          }, ...(0, _server.termQuery)(_elasticsearch_fieldnames.METRICSET_NAME, 'app')]
        }
      },
      aggs: {
        timeseriesData: {
          date_histogram: (0, _metrics.getMetricsDateHistogramParams)({
            start,
            end,
            metricsInterval: config.metricsInterval
          }),
          aggs
        },
        ...aggs
      }
    }
  };
  const {
    aggregations
  } = await apmEventClient.search('get_compute_usage', params);
  const timeseriesData = aggregations === null || aggregations === void 0 ? void 0 : aggregations.timeseriesData;
  return {
    title: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.computeUsage', {
      defaultMessage: 'Compute usage'
    }),
    key: 'compute_usage',
    yUnit: 'number',
    description: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.computeUsage.description', {
      defaultMessage: "Compute usage (in GB-seconds) is the execution time multiplied by the available memory size of your function's instances. The compute usage is a direct indicator for the costs of your serverless function."
    }),
    series: !timeseriesData || timeseriesData.buckets.length === 0 ? [] : [{
      title: _i18n.i18n.translate('xpack.apm.agentMetrics.serverless.computeUsage', {
        defaultMessage: 'Compute usage'
      }),
      key: 'compute_usage',
      type: 'bar',
      overallValue: calculateComputeUsageGBSeconds({
        faasBilledDuration: aggregations === null || aggregations === void 0 ? void 0 : aggregations.avgFaasBilledDuration.value,
        totalMemory: aggregations === null || aggregations === void 0 ? void 0 : aggregations.avgTotalMemory.value
      }),
      color: _uiTheme.euiLightVars.euiColorVis0,
      data: timeseriesData.buckets.map(bucket => {
        const computeUsage = calculateComputeUsageGBSeconds({
          faasBilledDuration: bucket.avgFaasBilledDuration.value,
          totalMemory: bucket.avgTotalMemory.value
        });
        return {
          x: bucket.key,
          y: computeUsage
        };
      })
    }]
  };
}