"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMlModelTypesForModelConfig = exports.getMlModelConfigsForModelIds = exports.fetchPipelineProcessorInferenceData = exports.fetchMlInferencePipelineProcessors = exports.fetchMlInferencePipelineProcessorNames = exports.fetchAndAddTrainedModelData = void 0;

var _data_frame_analytics = require("../../../../ml/common/constants/data_frame_analytics");

var _pipelines = require("../../../common/types/pipelines");

var _ml_inference_pipeline_utils = require("../../utils/ml_inference_pipeline_utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const fetchMlInferencePipelineProcessorNames = async (client, indexName) => {
  try {
    const mlInferencePipelineName = (0, _ml_inference_pipeline_utils.getInferencePipelineNameFromIndexName)(indexName);
    const {
      [mlInferencePipelineName]: {
        processors: mlInferencePipelineProcessors = []
      }
    } = await client.ingest.getPipeline({
      id: mlInferencePipelineName
    });
    return mlInferencePipelineProcessors.map(obj => {
      var _obj$pipeline;

      return (_obj$pipeline = obj.pipeline) === null || _obj$pipeline === void 0 ? void 0 : _obj$pipeline.name;
    }).filter(name => name !== undefined);
  } catch (err) {
    // The GET /_ingest/pipeline API returns an empty object on 404 Not Found. If someone provides
    // a bad index name, catch the error and return an empty array of names.
    return [];
  }
};

exports.fetchMlInferencePipelineProcessorNames = fetchMlInferencePipelineProcessorNames;

const fetchPipelineProcessorInferenceData = async (client, mlInferencePipelineProcessorNames) => {
  const mlInferencePipelineProcessorConfigs = await client.ingest.getPipeline({
    id: mlInferencePipelineProcessorNames.join()
  });
  return Object.keys(mlInferencePipelineProcessorConfigs).reduce((pipelineProcessorData, pipelineProcessorName) => {
    var _inferenceProcessor$i;

    // Get the processors for the current pipeline processor of the ML Inference Processor.
    const subProcessors = mlInferencePipelineProcessorConfigs[pipelineProcessorName].processors || []; // Find the inference processor, which we can assume there will only be one.

    const inferenceProcessor = subProcessors.find(obj => obj.hasOwnProperty('inference'));
    const trainedModelName = inferenceProcessor === null || inferenceProcessor === void 0 ? void 0 : (_inferenceProcessor$i = inferenceProcessor.inference) === null || _inferenceProcessor$i === void 0 ? void 0 : _inferenceProcessor$i.model_id;
    if (trainedModelName) pipelineProcessorData.push({
      modelState: _pipelines.TrainedModelState.NotDeployed,
      pipelineName: pipelineProcessorName,
      trainedModelName,
      types: []
    });
    return pipelineProcessorData;
  }, []);
};

exports.fetchPipelineProcessorInferenceData = fetchPipelineProcessorInferenceData;

const getMlModelTypesForModelConfig = trainedModel => {
  var _trainedModel$tags;

  if (!trainedModel) return [];
  const isBuiltIn = (_trainedModel$tags = trainedModel.tags) === null || _trainedModel$tags === void 0 ? void 0 : _trainedModel$tags.includes(_data_frame_analytics.BUILT_IN_MODEL_TAG);
  return [trainedModel.model_type, ...Object.keys(trainedModel.inference_config || {}), ...(isBuiltIn ? [_data_frame_analytics.BUILT_IN_MODEL_TAG] : [])].filter(type => type !== undefined);
};

exports.getMlModelTypesForModelConfig = getMlModelTypesForModelConfig;

const getMlModelConfigsForModelIds = async (client, trainedModelNames) => {
  const [trainedModels, trainedModelsStats] = await Promise.all([client.ml.getTrainedModels({
    model_id: trainedModelNames.join()
  }), client.ml.getTrainedModelsStats({
    model_id: trainedModelNames.join()
  })]);
  const modelConfigs = {};
  trainedModels.trained_model_configs.forEach(trainedModelData => {
    const trainedModelName = trainedModelData.model_id;

    if (trainedModelNames.includes(trainedModelName)) {
      modelConfigs[trainedModelName] = {
        modelState: _pipelines.TrainedModelState.NotDeployed,
        pipelineName: '',
        trainedModelName,
        types: getMlModelTypesForModelConfig(trainedModelData)
      };
    }
  });
  trainedModelsStats.trained_model_stats.forEach(trainedModelStats => {
    var _trainedModelStats$de;

    const trainedModelName = trainedModelStats.model_id;

    if (modelConfigs.hasOwnProperty(trainedModelName)) {
      var _trainedModelStats$de2;

      let modelState;

      switch ((_trainedModelStats$de = trainedModelStats.deployment_stats) === null || _trainedModelStats$de === void 0 ? void 0 : _trainedModelStats$de.state) {
        case 'started':
          modelState = _pipelines.TrainedModelState.Started;
          break;

        case 'starting':
          modelState = _pipelines.TrainedModelState.Starting;
          break;

        case 'stopping':
          modelState = _pipelines.TrainedModelState.Stopping;
          break;
        // @ts-ignore: type is wrong, "failed" is a possible state

        case 'failed':
          modelState = _pipelines.TrainedModelState.Failed;
          break;

        default:
          modelState = _pipelines.TrainedModelState.NotDeployed;
          break;
      }

      modelConfigs[trainedModelName].modelState = modelState;
      modelConfigs[trainedModelName].modelStateReason = (_trainedModelStats$de2 = trainedModelStats.deployment_stats) === null || _trainedModelStats$de2 === void 0 ? void 0 : _trainedModelStats$de2.reason;
    }
  });
  return modelConfigs;
};

exports.getMlModelConfigsForModelIds = getMlModelConfigsForModelIds;

const fetchAndAddTrainedModelData = async (client, pipelineProcessorData) => {
  const trainedModelNames = Array.from(new Set(pipelineProcessorData.map(pipeline => pipeline.trainedModelName)));
  const modelConfigs = await getMlModelConfigsForModelIds(client, trainedModelNames);
  return pipelineProcessorData.map(data => {
    const model = modelConfigs[data.trainedModelName];

    if (!model) {
      return data;
    }

    const {
      types,
      modelState,
      modelStateReason
    } = model;
    return { ...data,
      types,
      modelState,
      modelStateReason
    };
  });
};

exports.fetchAndAddTrainedModelData = fetchAndAddTrainedModelData;

const fetchMlInferencePipelineProcessors = async (client, indexName) => {
  const mlInferencePipelineProcessorNames = await fetchMlInferencePipelineProcessorNames(client, indexName); // Elasticsearch's GET pipelines API call will return all of the pipeline data if no ids are
  // provided. If we didn't find pipeline processors, return early to avoid fetching all of
  // the possible pipeline data.

  if (mlInferencePipelineProcessorNames.length === 0) return [];
  const pipelineProcessorInferenceData = await fetchPipelineProcessorInferenceData(client, mlInferencePipelineProcessorNames); // Elasticsearch's GET trained models and GET trained model stats API calls will return the
  // data/stats for all of the trained models if no ids are provided. If we didn't find any
  // inference processors, return early to avoid fetching all of the possible trained model data.

  if (pipelineProcessorInferenceData.length === 0) return [];
  const pipelines = await fetchAndAddTrainedModelData(client, pipelineProcessorInferenceData); // Due to restrictions with Kibana spaces we do not want to return the trained model name
  // to the UI. So we remove it from the data structure here.

  return pipelines.map(({
    trainedModelName,
    ...pipeline
  }) => pipeline);
};

exports.fetchMlInferencePipelineProcessors = fetchMlInferencePipelineProcessors;