"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmTransactionDurationTransformGenerator = void 0;

var _constants = require("../../../assets/constants");

var _slo_transform_template = require("../../../assets/transform_templates/slo_transform_template");

var _models = require("../../../types/models");

var _schema = require("../../../types/schema");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const APM_SOURCE_INDEX = 'metrics-apm*';

class ApmTransactionDurationTransformGenerator {
  getTransformParams(slo, spaceId) {
    if (!_models.apmTransactionDurationSLOSchema.is(slo)) {
      throw new Error(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }

    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildSource(slo), this.buildDestination(slo, spaceId), this.buildGroupBy(), this.buildAggregations(slo));
  }

  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id);
  }

  buildSource(slo) {
    const queryFilter = [];

    if (slo.indicator.params.service !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.name': slo.indicator.params.service
        }
      });
    }

    if (slo.indicator.params.environment !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.environment': slo.indicator.params.environment
        }
      });
    }

    if (slo.indicator.params.transaction_name !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.name': slo.indicator.params.transaction_name
        }
      });
    }

    if (slo.indicator.params.transaction_type !== _schema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.type': slo.indicator.params.transaction_type
        }
      });
    }

    return {
      index: APM_SOURCE_INDEX,
      runtime_mappings: {
        'slo.id': {
          type: 'keyword',
          script: {
            source: `emit('${slo.id}')`
          }
        }
      },
      query: {
        bool: {
          filter: [{
            match: {
              'transaction.root': true
            }
          }, ...queryFilter]
        }
      }
    };
  }

  buildDestination(slo, spaceId) {
    return {
      pipeline: (0, _constants.getSLOIngestPipelineName)(spaceId),
      index: (0, _constants.getSLODestinationIndexName)(spaceId)
    };
  }

  buildGroupBy() {
    return {
      'slo.id': {
        terms: {
          field: 'slo.id'
        }
      },
      '@timestamp': {
        date_histogram: {
          field: '@timestamp',
          calendar_interval: '1m'
        }
      },
      'slo.context.transaction.name': {
        terms: {
          field: 'transaction.name'
        }
      },
      'slo.context.transaction.type': {
        terms: {
          field: 'transaction.type'
        }
      },
      'slo.context.service.name': {
        terms: {
          field: 'service.name'
        }
      },
      'slo.context.service.environment': {
        terms: {
          field: 'service.environment'
        }
      }
    };
  }

  buildAggregations(slo) {
    const truncatedThreshold = Math.trunc(slo.indicator.params['threshold.us']);
    return {
      _numerator: {
        range: {
          field: 'transaction.duration.histogram',
          ranges: [{
            to: truncatedThreshold
          }]
        }
      },
      'slo.numerator': {
        bucket_script: {
          buckets_path: {
            numerator: `_numerator['*-${truncatedThreshold}.0']>_count`
          },
          script: 'params.numerator'
        }
      },
      'slo.denominator': {
        value_count: {
          field: 'transaction.duration.histogram'
        }
      }
    };
  }

}

exports.ApmTransactionDurationTransformGenerator = ApmTransactionDurationTransformGenerator;