""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import json
from helpers.logger import Logger
from helpers.utils import Utilities
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from framework.utility.task_utils import TaskUtilities
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE, REPO_URL

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class Initialize(Tasks):
    TASK_STATUS_MSG = "Initialize variables"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.task_utilities = TaskUtilities()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)
        self.utilities = Utilities()

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        # We must calculate this value before installing/updating any cyops RPM
        b_initialize = self._initialize()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        return b_initialize

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
    
    def _initialize(self):
        try:
            os.environ["LC_ALL"] = "en_US.UTF-8"
            product_yum_server = REPO_URL
            s_release_version = self.target_upgrade_version
            s_pkg = "cyops-common"

            s_nodejs_ver = "20"
            s_rabbitmq_intermediate_version_1="3.11.28"
            s_rabbitmq_intermediate_version_2="3.12.13"
            d_crypto_policies_back_ends = "/etc/crypto-policies/back-ends"

            # l_ssh_config_files="openssh.config \ libssh.config \ nss.config \ opensshserver.config"
            l_ssh_config_files = [
                "openssh.config", "libssh.config", "nss.config", "opensshserver.config"]

            flag_is_upgrade_supported = False
            s_repo_update_rpm = "/opt/fsr-elevate/elevate/cyops-repo-update-{}.el9.x86_64.rpm".format(
                s_release_version)
            s_url = product_yum_server
            d_yum_repo = "/etc/yum.repos.d"
            i_one_mb = 1024
            i_one_gb = 1048576
            i_free_space_required_in_boot = 40000
            i_column_width = 20
            f_cyops_release = "/etc/cyops-release"
            f_os_release = "/etc/os-release"
            l_supported_os = ["Rocky Linux", "Red Hat Enterprise Linux"]
            flag_is_rhel = False
            s_rhel_optional_rpms = "rhel-8-server-optional-rpms"

            # The "flag_lvm_creation_mandatory" checks if LVM creation is mandatory for FortiSOAR upgradation using upgrade script.
            # The variable value false: LVM creation is not mandatory. if LVMs are not present as per sizing guide, add warning message in logs and continue with the upgrade process.
            # The variable value true:  LVM creation is mandatory. if LVMs are not present as per sizing guide, exit the upgrade process.
            flag_lvm_creation_mandatory = False

            flag_is_enterprise = False
            flag_is_secure_message_exchange = False
            flag_is_forticloud_enterprise = False
            d_home_csadmin = "/home/csadmin"
            f_bash_profile = "{}/.bash_profile".format(d_home_csadmin)
            f_file_descriptor_3 = "/dev/fd/3"
            pem_file_backup_dir = "{}/pem_file_backup".format(d_home_csadmin)
            d_pgsql_path = "/var/lib/pgsql"
            f_pgsql_resv_space_file = "resv_space"

            # The "i_custom_yum_url" for user's offline repo.
            # The variable value 0: not exported the offline repo.
            # The variable value 1: exported the offline repo.
            i_custom_yum_url = 0
            f_etc_environment = "/etc/environment"

            # Declare a variable to display the post_upgrade failed messages
            var_post_upgrade_msg = []

            # The i_require_vol_space value is in block size.The value is 500 MB
            i_require_vol_space = 500000

            # The i_skip_tmp_validation added for automation
            # 0: validate the space requirement for /tmp and /var/tmp
            # 1: skip the space validation for /tmp and /var/tmp
            i_skip_tmp_validation = 0

            f_csadm_suggestor = "/opt/cyops/scripts/csadm/auto_completion/csadm_auto_completion_suggestor.py"

            cmd = 'python3 /opt/cyops/scripts/csadm/auto_completion/csadm_auto_completion_suggestor.py --get-all-names-for-current-flavor'
            output = self.cmd_line_utilities.execute_cmd(cmd, True)
            if output['std_out']:
                std_out = output['std_out']
                all_names_for_current_flavor = list((element.strip())
                                                    for element in (std_out.split('\n')) if element)

            s_yum_repo = product_yum_server
            
            custom_yum_url = self.config['custom_yum_url'] if 'custom_yum_url' in self.config and self.config['custom_yum_url'] else ""
            s_flag = "-Isk" if 'ignore_ssl_check' in self.config and self.config['ignore_ssl_check'] else "-Is"
            
            if 'ignore_ssl_check' in self.config and self.config['ignore_ssl_check']:
                # flag_ignore_ssl_check_for_repo in /etc/environment file
                # this file is used in postgresql/rpm/config.sh to download the postgres extension in case of custom yum repo does not have ssl certs
                with open(f_etc_environment, "a") as file:
                    file.write("flag_ignore_ssl_check_for_repo=true\n")
            
            # The s_url variable value will replace if the user provides the custom_yum_url. So s_old_yum_repo variable initiates with "repo.fortisoar.fortinet.com".
            # The default value of s_url is "repo.fortisoar.fortinet.com" in the script.
            if custom_yum_url:
                # Removing 'https://' or 'http://' from custom_yum_url as it is handled wherever it is needed.
                # At some places custom_yum_url is needed without 'https://' or 'http://'
                if custom_yum_url.startswith('https://'):
                    custom_yum_url=custom_yum_url[len('https://'):]
    
                if custom_yum_url.startswith('http://'):
                    custom_yum_url=custom_yum_url[len('http://'):]
            
                i_custom_yum_url = 1
                s_old_yum_repo = s_yum_repo
                s_yum_repo = custom_yum_url
            else:
                s_old_yum_repo = ""

            l_upgrade_supported_from_versions = []
            json_file_url = "https://" + f"{s_yum_repo}/{self.current_version}/upgrade_support_versions.json"
            response = self.utilities.send_get_request(json_file_url)
            status = response[0]
            if status == 200:
                json_object = json.loads(response[1].decode())
                for version in json_object:
                    if self.target_upgrade_version in json_object[version]:
                        l_upgrade_supported_from_versions.append(version)
            else:
                self.logger.error(response[1])
                self.print_txt(response[1])
            
            s_integrations_pip_conf="/opt/cyops-integrations/.env/pip.conf"
            s_workflow_pip_conf="/opt/cyops-workflow/.env/pip.conf"
            l_pip_conf=[s_integrations_pip_conf, s_workflow_pip_conf]
            d_crypto_policies_back_ends = "/etc/crypto-policies/back-ends"
            
            is_postgres_upgrade_available = self.config['is_postgres_upgrade_available'] if 'is_postgres_upgrade_available' in self.config and self.config['is_postgres_upgrade_available'] else False
            
            self.store_result_to_json(self.store_result_path, "pre-upgrade", "initialize", None, {
                'product_yum_server': product_yum_server,
                's_release_version': s_release_version,
                's_flag': s_flag,
                's_pkg': s_pkg,
                'l_upgrade_supported_from_versions': l_upgrade_supported_from_versions,
                's_nodejs_ver': s_nodejs_ver,
                's_rabbitmq_intermediate_version_1': s_rabbitmq_intermediate_version_1,
                's_rabbitmq_intermediate_version_2': s_rabbitmq_intermediate_version_2,
                'd_crypto_policies_back_ends': d_crypto_policies_back_ends,
                'l_ssh_config_files': l_ssh_config_files,
                'flag_is_upgrade_supported': flag_is_upgrade_supported,
                's_repo_update_rpm': s_repo_update_rpm,
                's_url': s_url,
                'd_yum_repo': d_yum_repo,
                'i_one_mb': i_one_mb,
                'i_one_gb': i_one_gb,
                'i_free_space_required_in_boot': i_free_space_required_in_boot,
                'i_column_width': i_column_width,
                'f_cyops_release': f_cyops_release,
                'f_os_release': f_os_release,
                'l_supported_os': l_supported_os,
                'flag_is_rhel': flag_is_rhel,
                's_rhel_optional_rpms': s_rhel_optional_rpms,
                'flag_lvm_creation_mandatory': flag_lvm_creation_mandatory,
                'flag_is_enterprise': flag_is_enterprise,
                'flag_is_secure_message_exchange': flag_is_secure_message_exchange,
                'flag_is_forticloud_enterprise': flag_is_forticloud_enterprise,
                'd_home_csadmin': d_home_csadmin,
                'f_bash_profile': f_bash_profile,
                'f_file_descriptor_3': f_file_descriptor_3,
                'pem_file_backup_dir': pem_file_backup_dir,
                'd_pgsql_path': d_pgsql_path,
                'f_pgsql_resv_space_file': f_pgsql_resv_space_file,
                'i_custom_yum_url': i_custom_yum_url,
                'f_etc_environment': f_etc_environment,
                'var_post_upgrade_msg': var_post_upgrade_msg,
                'i_require_vol_space': i_require_vol_space,
                'i_skip_tmp_validation': i_skip_tmp_validation,
                'f_csadm_suggestor': f_csadm_suggestor,
                'all_names_for_current_flavor': all_names_for_current_flavor,
                's_yum_repo': s_yum_repo,
                'custom_yum_url': custom_yum_url,
                's_old_yum_repo': s_old_yum_repo,
                'l_pip_conf' : l_pip_conf,
                'd_crypto_policies_back_ends' : d_crypto_policies_back_ends,
                'is_postgres_upgrade_available' : is_postgres_upgrade_available
            })
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(f"Exception occurred at initialization task. Refer logs at '{LOG_FILE}'")
