""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class BackupIntegrationTrustStore(Tasks):
    TASK_STATUS_MSG = "Backup of integration truststore"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._backup_integration_truststore()
        
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _remove_config_vm_call_from_bash_profile(self):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_bash_profile = step_result['f_bash_profile']
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))

    # The backup_integration_truststore() function is to take the backup of the
    # certificate pem file for integration sdk.
    def _backup_integration_truststore(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            pem_file_backup_dir = step_result['pem_file_backup_dir']
            cmd = "/opt/cyops-integrations/.env/bin/python3 -c 'import certifi; print (certifi.where())'"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            pem_file = result['std_out']
            if not pem_file:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                report_msg = "Failed to find the certificate pem file for integration sdk"
                print(report_msg)
                # not failing upgrade here
                return
            if '\n' in str(pem_file):
                pem_file = pem_file.replace('\n', '')
            if not os.path.exists(pem_file_backup_dir):
                os.mkdir(pem_file_backup_dir)
            os.system('cp {} {}'.format(pem_file, pem_file_backup_dir))
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            return
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at backup integration truststore task. Refer logs at '{LOG_FILE}'"
            )
