""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class HandleUpgradeEnterprise(Tasks):
    TASK_STATUS_MSG = "Handle post upgrade enterprise"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._handle_post_upgrade_enterprise()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
    
    # The post_upgrade() function run the post upgrade task for FortiSOAR enterprise.
    def _handle_post_upgrade_enterprise(self):
        try:
            # restart all services
            cmd = "csadm services --restart"
            self.cmd_line_utilities.execute_cmd(cmd)

            widget_install_msg = None
            widget_err_msg = None
            insert_rule_engine_msg = None
            rule_engine_err_msg = None
            
            # Install/Upgrade default widgets
            cmd = "sudo -u nginx php /opt/cyops-api/bin/console app:import:from:file -f /opt/cyops-api/src/DataFixtures/Widgets/widgets.json"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result['return_code'] != 0:
                if result['std_err']:
                    err_msg = f"ERROR: Failed to install/upgrade default widgets. Following error occurred =>\n{result['std_err']}"
                    self.logger.error(err_msg)
                widget_err_msg = "Failed to install/upgrade default widgets"
            else:
                widget_install_msg = "Installing/Upgrading default widgets is done"

            cmd = "sudo -u nginx /opt/cyops-workflow/.env/bin/python /opt/cyops-workflow/sealab/manage.py rules_engine_upgrade_data_fixture"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            if result['return_code'] != 0:
                if result['std_err']:
                    rule_engine_err_msg = f"ERROR: Failed to insert rules engine default data fixture. Following error occurred =>\n{result['std_err']}"
                    self.logger.error(rule_engine_err_msg)
                rule_engine_err_msg =  "Failed to execute to insert rules engine default data fixture."
            else:
                insert_rule_engine_msg = "Inserting rules engine default data fixture is done"
            
            if widget_install_msg and insert_rule_engine_msg:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])  
            elif widget_err_msg and not rule_engine_err_msg:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(widget_err_msg)
            elif rule_engine_err_msg and not widget_err_msg:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(rule_engine_err_msg)
            else:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(widget_err_msg)
                print(rule_engine_err_msg)
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at post upgrade enterprise task. Refer logs at '{LOG_FILE}'"
            )