""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import re
import requests
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

"""This task should be executed at the end of the pre-upgrade phase"""

class UpgradeFsrElevate(Tasks):
    PACKAGE_NAME = 'fsr-elevate'
    TASK_STATUS_MSG = f"Upgrade '{PACKAGE_NAME}'"
    BASE_URL = "https://{}/{}/fortisoar/x86_64"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        self.server_repo = step_result['s_yum_repo']
    
    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        result = self._update_fsr_elevate()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        return result

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
        
    def _update_fsr_elevate(self):
        build_number = self._get_fsr_elevate_latest_build_number()
        if build_number != 0:
            base_url = self.BASE_URL.format(self.server_repo,self.target_upgrade_version)
            cmd = f"yum update -y {base_url}/{self.PACKAGE_NAME}-{self.target_upgrade_version}-{build_number}.el9.x86_64.rpm"
        else:
            return False
        result = self.cmd_line_utilities.execute_cmd(cmd)
        return_code = result['return_code']
        err = result['std_err']
        if return_code != 0:
            if err:
                self.logger.error(f"ERROR: {err}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print("{}ERROR{}: Failed to upgrade '{}'. Please refer '{}' file.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],self.PACKAGE_NAME,LOG_FILE))
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            return False
            
        self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        return True
        
    def _get_fsr_elevate_latest_build_number(self) -> int:    
        base_url = self.BASE_URL.format(self.server_repo,self.target_upgrade_version)
        
        # Fetch the HTML content of the directory listing
        response = requests.get(base_url)
        html_content = response.text

        # Extract all RPM filenames matching the pattern
        pattern = re.compile(r'fsr-elevate-[0-9]+\.[0-9]+\.[0-9]+-[0-9]+\.el9\.x86_64\.rpm')
        matches = pattern.findall(html_content)

        # Extract the build numbers from the filenames
        build_numbers = [int(re.search(r'fsr-elevate-[0-9]+\.[0-9]+\.[0-9]+-([0-9]+)\.el9\.x86_64\.rpm', match).group(1)) for match in matches]
        
        if build_numbers:
            return max(build_numbers)
        print("No build number found at the specified repository URL")
        return 0
