""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class RevertRepoConfChanges(Tasks):
    TASK_STATUS_MSG = "Revert repo conf changes"
    LOCAL_CONFIG_DIR = "/opt/cyops/configs/fsr-elevate/local_download"
    LOCAL_PACKAGES_CONFIG_FILE = f"{LOCAL_CONFIG_DIR}/local_download.conf"
    LOCAL_YUM_CONFIG_FILE = f"{LOCAL_CONFIG_DIR}/local_yum.conf"
    LOCAL_PACKAGES_DOWNLOAD_DIR = "/opt/cyops/packages"
    LOCAL_PACKAGES_DIR = "fsr-packages"
    LOCAL_SYSTEM_CONNECTORS_DIR = "fsr-connectors"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'upgrade'

    def get_description(self) -> str:
        return "This task will manage packages like nodejs etc."

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._revert_repo_conf_changes()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _revert_repo_conf_changes(self):
        try:
            step_result = self.get_step_results('upgrade', 'upgrade')
            flag_upgrade_from_local = step_result['flag_upgrade_from_local']
            local_download_dir = step_result['local_download_dir']

            self._update_yum_repository_conf_files("revert_bkp")  

            if flag_upgrade_from_local:
                # remove the local package dir
                os.system(f"chattr -iR {local_download_dir}")
                os.system(f"rm -rf {local_download_dir}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at revert repo conf changes task. Refer logs at '{LOG_FILE}'"
            )

    
    def _update_yum_repository_conf_files(self, action):
        step_result = self.get_step_results('upgrade', 'upgrade')
        local_download_dir = step_result['local_download_dir']
        
        repo_conf_file_dir = "/etc/yum.repos.d"
        fsr_local_repo_file = "fsr-local.repo"
        conf_file_list = ["fsr-app.repo","fsr-os.repo","fsr-third-party.repo"]

        local_repo_config_lines_list = [
            "[fsr-local]\n",
            "name=FortiSOAR Local Repository\n",
            f"baseurl={local_download_dir}/{self.LOCAL_PACKAGES_DIR}\n",
            "gpgcheck=0\n",
            "enabled=1\n\n"

            "[fsr-connectors]\n",
            "name=FortiSOAR Connectors Local Repository\n"
            f"baseurl={local_download_dir}/{self.LOCAL_SYSTEM_CONNECTORS_DIR}\n",
            "gpgcheck=0\n",
            "enabled=1\n\n"
        ]

        local_yum_config_lines_list=[
            "[main]\n",
            "gpgcheck=1\n",
            "installonly_limit=3\n",
            "clean_requirements_on_remove=True\n",
            "best=True\n",
            "skip_if_unavailable=False\n",
            "metadata_expire=1800s\n",
            "keepcache=1\n", # keep the local downloaded rpms
        ]

        if action=="create_bkp":
            # create the backup of the fsr yum repository conf files
            for conf_file in conf_file_list:
                os.system(f"mv  {repo_conf_file_dir}/{conf_file} {repo_conf_file_dir}/{conf_file}-bkp") 

            # create a local .repo file pointing to the local dir where the packages are downloaded
            with open(f"{repo_conf_file_dir}/{fsr_local_repo_file}", "w") as file:
                file.writelines(local_repo_config_lines_list)        

        elif action=="revert_bkp":
            # check if local .repo file exists 
            if os.path.exists(f"{repo_conf_file_dir}/{fsr_local_repo_file}"):
                for conf_file in conf_file_list:
                    os.system(f"mv  {repo_conf_file_dir}/{conf_file}-bkp {repo_conf_file_dir}/{conf_file}") 
                # remove the local .repo file
                os.remove(f"{repo_conf_file_dir}/{fsr_local_repo_file}")    
        elif action=="create_local_yum_conf":
            # create a local yum conf file
            with open(f"{self.LOCAL_YUM_CONFIG_FILE}", "w") as file:
                file.writelines(local_yum_config_lines_list) 

        else:
            print("Invalid argument!")    
