""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import yaml
import requests
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

try:
    sys.path.append(os.path.abspath("/opt/cyops/scripts/.lib"))
    from PasswordModule import decrypt
except ImportError:
    pass

GET_INDICES_URL = 'https://localhost:9200/_cat/indices?h=index'
DELETE_INDEX_URL = 'https://localhost:9200/{}'
GET_INDEX_DETAILS_URL = 'https://localhost:9200/{}/_settings'
CONFIG_FILES_YAML = '/opt/cyops/configs/database/db_config.yml'
ELASTICSEARCH_KEY = 'elasticsearch'
ES_USER = 'es_user'
ES_PASSWORD = 'secret'
ES_EXTERNAL_KEY = 'es_external'
COMPATIBILITY_AFTER = 7000000
TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class DeleteIncompatibleIndices(Tasks):
    TASK_STATUS_MSG = "Delete incompatible indices"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)
        self.es_config = {}
        self.es_session = None

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._delete_incompatible_indices()
        
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _get_es_config(self):
        with open(CONFIG_FILES_YAML, 'r') as db_config:
            config = {}
            config.update(yaml.safe_load(db_config))
        self.es_config = config.get(ELASTICSEARCH_KEY)

    def _is_es_external(self) -> bool:
        return self.es_config.get(ES_EXTERNAL_KEY)

    def _get_session(self) -> str:
        if not self.es_config:
            raise Exception("Failed to get ES config")
        password = decrypt(self.es_config.get(ES_PASSWORD))
        if not password.startswith('Password:'):
            raise Exception('Password decryption failed')
        password = password[len('Password:'):]
        user = self.es_config.get(ES_USER)
        session = requests.Session()
        session.auth = (user, password)
        return session

    def _get_incompatible_indices(self) -> list:
        try:
            response = self.es_session.get(GET_INDICES_URL, verify=False)
            if response.status_code != 200:
                return [response.status_code]
            indices = ' '.join(response.content.decode(
                'utf-8').split('\n')).split()
            incompatible_indices = []
            if indices:
                for index in indices:
                    index_details_response = self.es_session.get(GET_INDEX_DETAILS_URL.format(index), verify=False)
                    if index_details_response.status_code != 200:
                        return [index_details_response.status_code]
                    index_details = index_details_response.json().get(index)
                    version_created = index_details.get("settings").get(
                        "index").get("version").get("created")
                    version_updated = index_details.get("settings").get(
                        "index").get("version").get("updated")
                    check_version = version_updated if version_updated else version_created
                    if int(check_version) <= COMPATIBILITY_AFTER:
                        incompatible_indices.append(index)
                return incompatible_indices
            return []
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check incompatible indices task. Refer logs at '{LOG_FILE}'"
            )
            return []

    def _delete_incompatible_indices(self) -> None:
        try:
            self._get_es_config()
            # In case of external ES do nothing
            if self._is_es_external():
                self.store_result_to_json(self.store_result_path, 'pre-upgrade',
                                        'delete_incompatible_indices', 'delete_incompatible_indices', [])

                report_msg = "ES is external."
                self.print_txt(report_msg)
                return
            self.es_session = self._get_session()

            indices = self._get_incompatible_indices()
            deleted_indices = []  # will only contain FSR indices

            if indices and len(indices) == 1 and indices[0] != 200:
                err_msg = "Unable to fetch Elasticsearch indices."
                incompatible_indices = {'a_deleted_indices': deleted_indices,
                                    'i_number_of_incompatible_indices': len(deleted_indices)}
                self.store_result_to_json(self.store_result_path, 'pre-upgrade',
                                    'delete_incompatible_indices', 'delete_incompatible_indices', incompatible_indices)
                self.logger.error("ERROR: " + err_msg)
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                msg = "{}ERROR{}: {}".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],err_msg)
                print(msg)
                sys.exit()

            for index in indices:
                response = self.es_session.delete(DELETE_INDEX_URL.format(index), verify=False)
                if response.status_code == 200:
                    if index.startswith('cyops_'):
                        index = index[len("cyops_"):]
                        deleted_indices.append(index)
            
            if deleted_indices:
                report_msg = "Incompatible indices got collected successfully."
                self.logger.debug(
                    "Deleted indices are : {}".format(deleted_indices))
            else:
                report_msg = "No incompatible indices found."
            incompatible_indices = {'a_deleted_indices': deleted_indices,
                                    'i_number_of_incompatible_indices': len(deleted_indices)}
            self.store_result_to_json(self.store_result_path, 'pre-upgrade',
                                    'delete_incompatible_indices', 'delete_incompatible_indices', incompatible_indices)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check incompatible indices task. Refer logs at '{LOG_FILE}'"
            )
