""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class UpdateEnvironmentFile(Tasks):
    TASK_STATUS_MSG = "Add Sanitization to Nginx Access logs and UWSGI Logs"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        self._perform_log_sanitization()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
        
    def _perform_log_sanitization(self):
        try: 
            # Define the Nginx configuration file paths
            f_nginx_conf = "/etc/nginx/nginx.conf"
            f_cyops_api_conf = "/etc/nginx/conf.d/cyops-api.conf"
            f_cyops_workflow_conf = "/etc/nginx/conf.d/cyops-workflow.conf"
            f_cyops_integrations_conf = "/etc/nginx/conf.d/cyops-integrations.conf"
            f_cyops_routing_agent_conf = "/etc/nginx/conf.d/cyops-routing-agent.conf"
            f_cyops_archival_conf = "/etc/nginx/conf.d/cyops-archival.conf"

            # Check if nginx.conf exists
            if os.path.exists(f_nginx_conf):
                # Check if the map directive exists
                with open(f_nginx_conf, "r") as file:
                    nginx_conf_content = file.read()

                if "map $request_uri $log_args" not in nginx_conf_content:
                    # Backup the original configuration file
                    os.system(f"cp {f_nginx_conf} {f_nginx_conf}.bkp")

                    # Insert the map directive before "log_format main"
                    with open(f_nginx_conf, "r") as file:
                        lines = file.readlines()

                    with open(f_nginx_conf, "w") as file:
                        for line in lines:
                            if "log_format  main" in line:
                                file.write("    map $request_uri $log_args {\n")
                                file.write("        default $request_uri;\n")
                                file.write("        ~^(.*)secretKey=[^&]*(.*)$ $1secretKey=*****$2;\n")
                                file.write("    }\n")
                            file.write(line)

                    # Replace "$request" with "$request_method $log_args"
                    with open(f_nginx_conf, "r") as file:
                        content = file.read()
                    content = content.replace('"$request"', '"$request_method $log_args"')
                    with open(f_nginx_conf, "w") as file:
                        file.write(content)
                    
                    print("Added new map directive in nginx.conf.")

                else:
                    print("Map directive already exists in Nginx configuration. Skipping changes.")
            else:
                print(f"File {f_nginx_conf} does not exist. Skipping modifications.")

            # List of files and their respective patterns to check and update
            files_to_update = [
                (f_cyops_api_conf, "/var/log/cyops/cyops-api/ssl_cyops_api_access.log"),
                (f_cyops_workflow_conf, "/var/log/cyops/cyops-workflow/ssl_cyops_workflow_access.log"),
                (f_cyops_integrations_conf, "/var/log/cyops/cyops-integrations/ssl_cyops_integrations_access.log"),
                (f_cyops_routing_agent_conf, "/var/log/cyops/cyops-routing-agent/ssl_cyops_routing_agent_access.log"),
                (f_cyops_archival_conf, "/var/log/cyops/cyops-archival/ssl_cyops_archival_access.log"),
            ]
            for file_path, log_path in files_to_update:
                if os.path.exists(file_path):
                    with open(file_path, "r") as file:
                        file_content = file.read()

                    # Check if the "main" log_format exists for the access_log directive
                    if f"access_log {log_path} main;" not in file_content:
                        file_content = file_content.replace(f"access_log {log_path};", f"access_log {log_path} main;")
                        with open(file_path, "w") as file:
                            file.write(file_content)
                    else:
                        print(f"Access log format already updated in {file_path}. Skipping changes.")
                else:
                    print(f"File {file_path} does not exist. Skipping modifications.")

            # Restart Nginx and handle errors
            if subprocess.run(["systemctl", "restart", "nginx"]).returncode != 0:
                print("Error: Failed to restart Nginx after configuration change. Restoring the original configuration.")
                if os.path.exists(f"{f_nginx_conf}.bkp"):
                    os.rename(f"{f_nginx_conf}.bkp", f_nginx_conf)
                subprocess.run(["systemctl", "restart", "nginx"])
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return

            # Update UWSGI configuration
            wsgi_files = [
                "/etc/uwsgi.d/archival_wsgi.ini",
                "/etc/uwsgi.d/integrations_wsgi.ini",
                "/etc/uwsgi.d/postman_wsgi.ini",
                "/etc/uwsgi.d/sealab_wsgi.ini",
            ]
            for wsgi_file in wsgi_files:
                if os.path.exists(wsgi_file):
                    with open(wsgi_file, "r") as file:
                        wsgi_content = file.read()

                    # Check if "disable-logging = True" is already present
                    if "disable-logging = True" not in wsgi_content:
                        with open(wsgi_file, "a") as file:
                            file.write("disable-logging = True\n")
                    else:
                        print(f"'disable-logging = True' already exists in {wsgi_file}. Skipping changes.")
                else:
                    print(f"File {wsgi_file} does not exist. Skipping modifications.")

            # Restart UWSGI and handle errors
            if subprocess.run(["systemctl", "restart", "uwsgi"]).returncode != 0:
                print("ERROR: Failed to restart UWSGI after configuration change.")
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return

            print("Nginx and UWSGI configuration updated and reloaded successfully.")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])

        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at Nginx Log Sanitization. Refer logs at '{LOG_FILE}'"
            )