""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os, sys
import yaml
import psycopg2
import json
from framework.base.tasks import Tasks
from helpers.logger import Logger
from constants import REPORT_FILE, STEP_RESULT_FILE, TASK_STATUS, TASK_LOG_STATUS, TEXT_COLOR, LOG_FILE


CONFIG_FILES_YAML = '/opt/cyops/configs/database/db_config.yml'
PSQL_SEARCH_KEY = 'postgres'
PG_PASS_KEY ='pg_password'
PG_HOST_KEY = 'pg_host'
PG_USER_KEY = 'pg_user'
PG_PORT_KEY = 'pg_port'
PG_EXTERNAL = 'pg_external'

try:
    sys.path.append(os.path.abspath("/opt/cyops/scripts/.lib"))
    from PasswordModule import decrypt
except ImportError:
    pass


class UpdateSystemSettings(Tasks):
    TASK_STATUS_MSG = "Update System Settings'"

    """Implementation class for tasks"""
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)
        self.psql_config = None

    @property
    def tags(self) -> str:
        """Tags for task"""
        return 'post_upgrade'

    
    def get_description(self) -> str:
        """Description of a method"""
        return ""

    
    def is_supported(self) -> bool:
        """To check task is for target upgrade version/platform"""
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version
    
    def _get_pgsql_config(self):
        with open(CONFIG_FILES_YAML, 'r') as db_config:
            config = {}
            config.update(yaml.safe_load(db_config))
        self.psql_config = config.get(PSQL_SEARCH_KEY)

    def _update_cyops_system_settings(self):
        if self.psql_config is None:
            self._get_pgsql_config()
        s_pg_pass = decrypt(self.psql_config.get(PG_PASS_KEY))
        if not s_pg_pass.startswith('Password:'):
            raise Exception('Password decryption failed')
        s_pg_pass = s_pg_pass[len('Password:'):]
        s_pg_user = self.psql_config.get(PG_USER_KEY)
        s_pg_host = self.psql_config.get(PG_HOST_KEY)
        i_pg_port = self.psql_config.get(PG_PORT_KEY)
        s_pg_db_name = "venom"

        DB_CONFIG = {
            "dbname": s_pg_db_name,
            "user": s_pg_user,
            "password": s_pg_pass,
            "host": s_pg_host,
            "port": i_pg_port
        }
        queries = [
            ("TAXII Server Details", json.dumps({"status": {"enabled": False}})),
            ("Restricted File Mime-types", json.dumps({"values": ["image/svg", "image/svg+xml"]})),
            ("Launch On Boot", json.dumps({"values": {"value": True}}))
        ]
        try:
            conn = psycopg2.connect(**DB_CONFIG)
            cur = conn.cursor()
            for name, json_value in queries:
                sql = """UPDATE cyops.system_settings SET publicvalues = %s WHERE name = %s"""
                cur.execute(sql, (json_value, name))
            conn.commit()
            cur.close()
            conn.close()
            print("Updates successful!")

        except psycopg2.Error as e:
            print(f"Database error: {e}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])

    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    
    def execute(self) -> None:
        """To execute main code"""
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._update_cyops_system_settings()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    
    def validate(self) -> bool:
        """Write validations in this function"""
        return True

    
    def _print_status_msg(self, msg:str, status:str) -> None:
        """Use this function to print status message of task execution"""
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
