""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import (
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE
)

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class ValidateOsCompatibility(Tasks):
    TASK_STATUS_MSG = "Validate operating system compatibility"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return "pre-upgrade"

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace(".", ""))
        target_upgrade_version = int(self.target_upgrade_version.replace(".", ""))
        return target_upgrade_version > current_version

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        b_verify_rhel_installation = self._verify_rhel_installation()
        b_verify_operating_system_compatibility = self._verify_operating_system_compatibility()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        
        return b_verify_rhel_installation and b_verify_operating_system_compatibility

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
    
    def _verify_rhel_installation(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            f_os_release = step_result["f_os_release"]
            with open(f_os_release, "r") as os_release:
                result = dict(
                    (a.strip(), (b.strip()))
                    for a, b in (
                        element.split("=")
                        for element in os_release.read().split("\n")
                        if element
                    )
                )
                os_flavor = result["NAME"].strip('"')
            step_result["s_current_operating_system"] = os_flavor
            self.store_result_to_json(
                self.store_result_path, "pre-upgrade", "initialize", None, step_result
            )
            os_rh = "Red Hat Enterprise Linux"
            if str(os_flavor) == str(os_rh):
                step_result["flag_is_rhel"] = True
                self.store_result_to_json(
                    self.store_result_path,
                    "pre-upgrade",
                    "initialize",
                    None,
                    step_result,
                )
                b_is_enabled = self._enable_rhel_server_optional_repo()
                if not b_is_enabled:
                    return False
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(f"Exception occurred at validate os compatibility task. Refer logs at '{LOG_FILE}'")

    # The following function check the s_rhel_optional_rpms repo enabled or not.
    # The s_rhel_optional_rpms repo require to install the extra packages on rhel instance.
    def _enable_rhel_server_optional_repo(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            s_rhel_optional_rpms = step_result["s_rhel_optional_rpms"]
            self.print_txt("Checking the {} repo...".format(s_rhel_optional_rpms))
            i_option_repo_status = os.system(
                "subscription-manager repos | grep -A3 's_rhel_optional_rpms' | grep -w 'Enabled' | awk '{print $2}'"
            )
            if i_option_repo_status and i_option_repo_status != 1:
                msg = "The {} is not enabled.".format(s_rhel_optional_rpms)
                self.print_txt(msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Enable rhel server optional repo",
                    {"result": False, "message": msg},
                )
                return False
            report_msg = "The {} is enabled.".format(s_rhel_optional_rpms)
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Enable rhel server optional repo",
                {"result": True, "message": report_msg},
            )
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(f"Exception occurred at validate os compatibility task. Refer logs at '{LOG_FILE}'")
            self.store_result_to_json(
                self.report_path,
                "Enable RHEL Server Optional Repo",
                {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )

    def _verify_operating_system_compatibility(self):
        try:
            flag_is_os_supported = False
            step_result = self.get_step_results("pre-upgrade", "initialize")
            s_current_operating_system = step_result["s_current_operating_system"]
            s_release_version = step_result["s_release_version"]
            l_upgrade_supported_from_versions = step_result[
                "l_upgrade_supported_from_versions"
            ]
            step_result["flag_is_os_supported"] = flag_is_os_supported

            # Loop through the list and grep each value against s_current_operating_system
            # Break the loop if found
            l_supported_os = step_result["l_supported_os"]
            for supported_os in l_supported_os:
                if s_current_operating_system == supported_os:
                    flag_is_os_supported = True
                    step_result["flag_is_os_supported"] = flag_is_os_supported
                    break
            self.store_result_to_json(
                self.store_result_path, "pre-upgrade", "initialize", None, step_result
            )

            # Exit with error if supported OS not found
            if not flag_is_os_supported:
                self._remove_config_vm_call_from_bash_profile()
                report_msg = "Operating System found : {}. Upgrade to {} is only available from version : {}. Supported operating systems are : {}".format(
                    s_current_operating_system,
                    s_release_version,
                    l_upgrade_supported_from_versions,
                    l_supported_os,
                )
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Verify Operating System Compatibility",
                    {"result": False, "message": report_msg},
                )
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(f"Non supported operating system detected : {s_current_operating_system}")
                self.print_txt("Supported operating systems are:")
                for supported_os in l_supported_os:
                    i_index = int(l_supported_os.index(supported_os)) + 1
                    self.print_txt("           {}. {}".format(i_index, supported_os))
                return False
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Verify Operating System Compatibility",
                {
                    "result": True,
                    "message": f"The current operating system is {s_current_operating_system}, which is supported for upgrade.",
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(f"Exception occurred at validate os compatibility task. Refer logs at '{LOG_FILE}'")
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Verify Operating System Compatibility",
                {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )

    def _remove_config_vm_call_from_bash_profile(self):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        step_result = self.get_step_results("pre-upgrade", "initialize")
        f_bash_profile = step_result["f_bash_profile"]
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))
