""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import math
import yaml
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import (
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE
)

try:
    sys.path.append(os.path.abspath("/opt/cyops/scripts/.lib"))
    from PasswordModule import decrypt
except ImportError:
    pass

TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}
CONFIG_FILES_YAML = '/opt/cyops/configs/database/db_config.yml'
PSQL_SEARCH_KEY = 'postgres'
PG_PASS_KEY ='pg_password'
PG_HOST_KEY = 'pg_host'
PG_USER_KEY = 'pg_user'
PG_PORT_KEY = 'pg_port'
PG_EXTERNAL = 'pg_external'

class CheckSpaceRequirement(Tasks):
    TASK_STATUS_MSG = "Validate required space available"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)
        self.psql_config = None

    @property
    def tags(self) -> str:
        return "pre-upgrade"

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace(".", ""))
        target_upgrade_version = int(self.target_upgrade_version.replace(".", ""))
        return target_upgrade_version > current_version

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        b_check_root_directory_free_space = self._check_root_directory_free_space()
        b_check_boot_directory_free_space = self._check_boot_directory_free_space()
        b_check_log_directory_free_space = self._check_log_directory_free_space()
        b_check_opt_directory_free_space = self._check_opt_directory_free_space()
        b_check_tmp_directory_free_space = self._check_tmp_directory_free_space()
        b_check_db_directory_free_space = self._check_workflow_data_free_space()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        
        if(b_check_root_directory_free_space and b_check_boot_directory_free_space
                and b_check_log_directory_free_space and b_check_opt_directory_free_space
                and b_check_tmp_directory_free_space and b_check_db_directory_free_space):
            return True
        return False

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _get_directory_space_requirement(self, dir, space_req_kb):
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        i_one_mb = step_result['i_one_mb']
        i_one_gb = step_result['i_one_gb']
        
        flag_is_required_space_available = True
        cmd = "df -k --total '{}'".format(dir)
        result = self.cmd_line_utilities.execute_cmd(cmd, True)
        std_out = result["std_out"].split("total")[1]
        l_elements = list(
            (element.strip()) for element in (std_out.split(" ")) if element
        )
        i_dir_free_space_kb = int(l_elements[2])
        
        additional_space_req_kb = None
        additional_space_req_msg = None
        
        if i_dir_free_space_kb < space_req_kb:
            flag_is_required_space_available = False
            additional_space_req_kb = space_req_kb - i_dir_free_space_kb
        
        i_dir_free_space_unit = self._convert_size(i_dir_free_space_kb, i_one_mb, i_one_gb)
        i_dir_free_space = i_dir_free_space_unit.split(" ")[0]
        unit = i_dir_free_space_unit.split(" ")[1]
        i_dir_free_space_unit = f"{math.ceil(float(i_dir_free_space))} {unit}"
        
        space_req_unit = self._convert_size(space_req_kb, i_one_mb, i_one_gb)
        space_req = space_req_unit.split(" ")[0]
        unit = space_req_unit.split(" ")[1]
        space_req_unit = f"{math.ceil(float(space_req))} {unit}"
        
        if additional_space_req_kb:
            additional_space_req_unit = self._convert_size(additional_space_req_kb, i_one_mb, i_one_gb)
            additional_space_req = additional_space_req_unit.split(" ")[0]
            unit = additional_space_req_unit.split(" ")[1]
            additional_space_req_unit = f"{math.ceil(float(additional_space_req))} {unit}"
        
        available_free_space_msg = f"Available Free Space on '{dir}' : {i_dir_free_space_unit}"
        recommended_free_space_msg = f"Required Free Space on '{dir}' : {space_req_unit}"
        if not flag_is_required_space_available:
            additional_space_req_msg = f"Do allocate an additional {additional_space_req_unit} into the '{dir}'"
            
        return flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg
        
    def _check_root_directory_free_space(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            i_one_gb = step_result["i_one_gb"]
            
            i_minimum_required_space_gb = 2 # 2 GB
            i_free_space_required_in_root_kb = int(i_minimum_required_space_gb) * int(i_one_gb) # 2 x 1048576 KB
            
            flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg = self._get_directory_space_requirement('/',i_free_space_required_in_root_kb)
            
            if not flag_is_required_space_available:
                report_msg = f"{available_free_space_msg}. {recommended_free_space_msg}. {additional_space_req_msg}."
                self.logger.error(report_msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Check '/' Directory Free Space",
                    {
                        "result": False,
                        "message": report_msg,
                    },
                )
                self._print_status_msg(self.TASK_STATUS_MSG+" in '/'", TASK_STATUS["FAILED"])
                print(f"{available_free_space_msg}. {recommended_free_space_msg}.\n{additional_space_req_msg}.")
                return False
            report_msg = "Required free space is available in '/'."
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/' Directory Free Space",
                {
                    "result": True,
                    "message": report_msg,
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/'", TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/'", TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/' Directory Free Space",
                {"result": True, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )
    
    # The check_boot() function check the /boot drive space is sufficient to upgrade or not.
    # If /boot space not sufficient, then the script exits.
    def _check_boot_directory_free_space(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            i_free_space_required_in_boot_kb = step_result["i_free_space_required_in_boot"] # 40000 Kb
            flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg = self._get_directory_space_requirement('/boot',i_free_space_required_in_boot_kb)
            if not flag_is_required_space_available:
                report_msg = f"{available_free_space_msg}. {recommended_free_space_msg}. {additional_space_req_msg}."
                self.logger.error(report_msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Check '/boot' Directory Free Space",
                    {
                        "result": False,
                        "message": report_msg,
                    },
                )
                self._print_status_msg(self.TASK_STATUS_MSG+" in '/boot'", TASK_STATUS["FAILED"])
                print(f"{available_free_space_msg}. {recommended_free_space_msg}.\n{additional_space_req_msg}.")
                return False
            report_msg = "Required free space is available in '/boot'."
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/boot' Directory Free Space",
                {
                    "result": True,
                    "message": report_msg,
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/boot'", TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/boot'", TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/boot' Directory Free Space",
                {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )
    
    def _check_log_directory_free_space(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            i_one_mb = step_result["i_one_mb"]
            i_free_space_required_in_log_mb = 200 # 200 Mb
            i_free_space_required_in_log_kb = i_one_mb * i_free_space_required_in_log_mb
            flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg = self._get_directory_space_requirement('/var/log',i_free_space_required_in_log_kb)
            if not flag_is_required_space_available:
                report_msg = f"{available_free_space_msg}. {recommended_free_space_msg}. {additional_space_req_msg}."
                self.logger.error(report_msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Check '/var/log' Directory Free Space",
                    {
                        "result": False,
                        "message": report_msg,
                    },
                )
                self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/log'", TASK_STATUS["FAILED"])
                print(f"{available_free_space_msg}. {recommended_free_space_msg}.\n{additional_space_req_msg}.")
                return False
            report_msg = "Required free space is available in '/var/log'."
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/var/log' Directory Free Space",
                {
                    "result": True,
                    "message": report_msg,
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/log'", TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/log'", TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/var/log' Directory Free Space",
                {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )
    
    def _check_opt_directory_free_space(self):
        try:
            step_result = self.get_step_results("pre-upgrade", "initialize")
            i_one_mb = step_result["i_one_mb"]
            i_free_space_required_in_opt_mb = 200 # 200 Mb
            i_free_space_required_in_opt_kb = i_one_mb * i_free_space_required_in_opt_mb
            flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg = self._get_directory_space_requirement('/opt',i_free_space_required_in_opt_kb)
            if not flag_is_required_space_available:
                report_msg = f"{available_free_space_msg}. {recommended_free_space_msg}. {additional_space_req_msg}."
                self.logger.error(report_msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Check '/opt' Directory Free Space",
                    {
                        "result": False,
                        "message": report_msg,
                    },
                )
                self._print_status_msg(self.TASK_STATUS_MSG+" in '/opt'", TASK_STATUS["FAILED"])
                print(f"{available_free_space_msg}. {recommended_free_space_msg}.\n{additional_space_req_msg}.")
                return False
            report_msg = "Required free space is available in '/opt'."
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/opt' Directory Free Space",
                {
                    "result": True,
                    "message": report_msg,
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/opt'", TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/opt'", TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/opt' Directory Free Space",
                {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"},
            )
        
    # The space_requirement(){ function checks the space requirement for /var/tmp
    # /var/tmp is the directory used by dracut to create the initramfs for new kernel during an upgrade
    def _check_tmp_directory_free_space(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            i_free_space_required_in_tmp_kb = step_result['i_require_vol_space'] # 500000 Kb
            flag_is_required_space_available, available_free_space_msg, recommended_free_space_msg, additional_space_req_msg = self._get_directory_space_requirement('/var/tmp',i_free_space_required_in_tmp_kb)
            if not flag_is_required_space_available:
                report_msg = f"{available_free_space_msg}. {recommended_free_space_msg}. {additional_space_req_msg}."
                self.logger.error(report_msg)
                self.store_result_to_json(
                    self.report_path,
                    None,
                    None,
                    "Check '/var/tmp' Directory Free Space",
                    {
                        "result": False,
                        "message": report_msg,
                    },
                )
                self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/tmp'", TASK_STATUS["FAILED"])
                print(f"{available_free_space_msg}. {recommended_free_space_msg}.\n{additional_space_req_msg}.")
                return False
            report_msg = "Required free space is available in '/var/tmp'."
            self.store_result_to_json(
                self.report_path,
                None,
                None,
                "Check '/var/tmp' Directory Free Space",
                {
                    "result": True,
                    "message": report_msg,
                },
            )
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/tmp'", TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG+" in '/var/tmp'", TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    "Check '/var/tmp' Directory Free Space", {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})

        # The space_requirement for workflow data: System should have 2X of free space to upgrade
        # if workflow data is 100GB then system should have 200GB of free space
    def _check_workflow_data_free_space(self):
        location = '/var/lib/pgsql'
        status = TASK_STATUS["DONE"]
        report_msg = f"Required free space is available in '{location}'."
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']

        if not flag_is_enterprise:
            self._print_status_msg(self.TASK_STATUS_MSG + " in " + location, TASK_STATUS["SKIPPED"])
            return True

        try:
            self._get_pgsql_config()
            i_current_size = self._get_workflow_database_size()
            is_external = self.psql_config.get(PG_EXTERNAL, False)

            i_current_size = i_current_size * 2
            if is_external:
                required_size = self.convert_byte_human_readable(i_current_size)
                report_msg = f"Validate required free space ({required_size}) is available on external database."
                status = TASK_STATUS["SKIPPED"]
            else:
                (is_space_available, available_space_msg, recommended_space_msg,
                 additional_space_msg) = self._get_directory_space_requirement(location, i_current_size)
                if not is_space_available:
                    report_msg = f"{available_space_msg}. {recommended_space_msg}. {additional_space_msg}."
                    self.logger.error(report_msg)
                    status = TASK_STATUS["FAILED"]

        except Exception as ex:
            status = TASK_STATUS["FAILED"]
            self.logger.exception("ERROR: {}".format(ex))
            report_msg = f"Exception occurred at check space requirement task. Refer logs at '{LOG_FILE}'"

        is_success = (status != TASK_STATUS["FAILED"])
        self.store_result_to_json(
            self.report_path, None, None, f"Check '{location}' Free Space",
            {"result": is_success,  "message": report_msg},
        )

        self._print_status_msg(self.TASK_STATUS_MSG + " in " + location, status)
        if status != TASK_STATUS["DONE"]:
            print(report_msg)

        return is_success

    def _get_pgsql_config(self):
        with open(CONFIG_FILES_YAML, 'r') as db_config:
            config = {}
            config.update(yaml.safe_load(db_config))
        self.psql_config = config.get(PSQL_SEARCH_KEY)

    def _get_workflow_database_size(self):
        if self.psql_config is None:
            self._get_pgsql_config()

        s_pg_pass = decrypt(self.psql_config.get(PG_PASS_KEY))
        if not s_pg_pass.startswith('Password:'):
            raise Exception('Password decryption failed')
        s_pg_pass = s_pg_pass[len('Password:'):]
        s_pg_user = self.psql_config.get(PG_USER_KEY)
        s_pg_host = self.psql_config.get(PG_HOST_KEY)
        i_pg_port = self.psql_config.get(PG_PORT_KEY)

        s_pg_db_name = "sealab"
        query = f"SELECT pg_database_size('{s_pg_db_name}') / 1024;"
        cmd = (
            f"psql 'postgresql://{s_pg_user}:{s_pg_pass}@{s_pg_host}:{i_pg_port}/{s_pg_db_name}' -t --no-align -c \"{query}\""
        )

        result = self.cmd_line_utilities.execute_cmd(cmd, True)
        wf_size = result["std_out"]
        return int(wf_size)

    def convert_byte_human_readable(self, size):
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        i_one_mb = step_result['i_one_mb']
        i_one_gb = step_result['i_one_gb']
        readable_size = self._convert_size(size, i_one_mb, i_one_gb)
        i_dir_free_space = readable_size.split(" ")[0]
        unit = readable_size.split(" ")[1]
        readable_size = f"{math.ceil(float(i_dir_free_space))} {unit}"
        return readable_size

    # Parameters
    # Size in KB
    # Returns:
    # Size in MB if size in kb is > 1024KB
    # Size in GB if size in mb is > 1024MB
    # Also appends "GB" or "MB" string
    def _convert_size(self, i_size, i_one_mb, i_one_gb):
        if i_size > i_one_gb:
            i_divider = i_one_gb
            s_unit_i_size = "GB"
        elif i_size > i_one_mb:
            i_divider = i_one_mb
            s_unit_i_size = "MB"
        i_size = "{} {}".format(round(i_size / i_divider, 2), s_unit_i_size)
        return i_size