""" Copyright start
  Copyright (C) 2008 - 2024 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import json
import subprocess
from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from helpers.logger import Logger
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class UpdateProductNameInCyopsRelease(Tasks):
    TASK_STATUS_MSG = "Update product name in /etc/cyops-release"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'upgrade'

    def get_description(self) -> str:
        return ""
    
    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version
            
    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        self._update_product_name_in_cyops_release()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        
    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
 
    # The following function adds product_name in /etc/cyops-release with default value 'fortisoar' if it is not already present.
    def _update_product_name_in_cyops_release(self):
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_cyops_release = step_result['f_cyops_release']
        default_name = "fortisoar"
        try:
            with open(f_cyops_release, 'r') as file:
                if any('product_name=' in line for line in file):
                    print("product_name is already defined in /etc/cyops-release. No further changes needed.")
                    return
            
            subprocess.run(['chattr', '-i', f_cyops_release], check=True)
            with open(f_cyops_release, 'a') as file:
                file.write(f'product_name={default_name}\n')
                print("Added product_name in /etc/cyops-release.")
        
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at {self.TASK_STATUS_MSG} task. Refer logs at '{LOG_FILE}'"
            )
        
        finally:
            try:
                subprocess.run(['chattr', '+i', f_cyops_release], check=True)
            except Exception as ex:
                err_msg = "ERROR: {}".format(ex)
                self.logger.exception(err_msg)
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(f"Exception occurred at {self.TASK_STATUS_MSG} task while reapplying immutability flag to /etc/cyops-release. Refer logs at '{LOG_FILE}'")