""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import re
import sys
import glob
import configparser
from helpers.logger import Logger
from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class Upgrade(Tasks):
    TASK_STATUS_MSG = "Upgrade RPM packages"
    ENVIRONMENT_FILE_STATUS_MSG = "Update 'environment' file for custom yum repo"
    LOCAL_CONFIG_DIR = "/opt/cyops/configs/fsr-elevate/local_download"
    LOCAL_PACKAGES_CONFIG_FILE = f"{LOCAL_CONFIG_DIR}/local_download.conf"
    LOCAL_YUM_CONFIG_FILE = f"{LOCAL_CONFIG_DIR}/local_yum.conf"
    LOCAL_PACKAGES_DOWNLOAD_DIR = "/opt/cyops/packages"
    LOCAL_PACKAGES_DIR = "fsr-packages"
    LOCAL_SYSTEM_CONNECTORS_DIR = "fsr-connectors"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        self._upgrade()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
    
    def _yes_no_user_input(self):
        yes_list = ["y","Y","Yes","yes","YES"]
        no_list = ["n","N","No","no","NO"]
        i_count=0
        while i_count <=2:
            user_input = input()
            if user_input in yes_list:
                return True
            elif user_input in no_list:
                return False
            else:
                print("You have provided an invalid input. Enter Yes or No")   
                i_count+=1 
        if  i_count > 2:
            print("Max retries reached, exiting the upgrade.")   
            
    def _upgrade_nginx(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_nginx_ver = step_result['s_nginx_ver']
            # Default nginx getting installed is of lower version
            # Installing from nginx:22 installs latest version which is bundled in FortiSOAR Rockylinux-9 AppStream Repository
            # For this to happen, need to reset and then enable required stream
            commands = ["dnf module reset -y nginx",
                        f"dnf module enable -y nginx:{s_nginx_ver}",
                        "dnf module list nginx",
                        "dnf update nginx -y"]
            return_code = self._execute_command_list(commands)
            if (return_code == -1):
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                self.print_txt("Nginx update failed")
                self._exit(1)
            self.print_txt("Nginx updated successfully")
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )
    
    def _upgrade(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_pkg = step_result['s_pkg']
            flag_is_enterprise = step_result['flag_is_enterprise']
            flag_is_secure_message_exchange = step_result['flag_is_secure_message_exchange']
            
            flag_upgrade_from_local = False
            local_download_dir=None
            self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'flag_upgrade_from_local', flag_upgrade_from_local)
            self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'local_download_dir', local_download_dir)

            if int(self.current_version.replace('.', '')) < 760 :
                self._upgrade_rabbitmq_server()

            if flag_is_enterprise:
                
                # Nginx upgrade requires nginx yum modules to be enabled. 
                # In upgrade using locally downloaded packages, 
                # downloaded packages in local directory does not preserve modular metadata. 
                # Hence we need to perform the nginx upgrade before we point to the local directory. 
                self._upgrade_nginx()
                
                # Point to local directory of downloaded packages for yum upgrade. 
                if os.path.exists(f"{self.LOCAL_PACKAGES_CONFIG_FILE}"):
                    local_download_config = configparser.ConfigParser()
                    local_download_config.read(self.LOCAL_PACKAGES_CONFIG_FILE)

                    # Only for the one version entry in the conf file
                    for section in local_download_config.sections():
                        upgrade_fortisoar_version=section
                    # Get the values from the config file
                    flag_upgrade_from_local=local_download_config.getboolean(upgrade_fortisoar_version,'flag_upgrade_from_local')
                    local_download_dir=local_download_config.get(upgrade_fortisoar_version,'local_download_dir')
                    
                    self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'flag_upgrade_from_local', flag_upgrade_from_local)
                    self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'local_download_dir', local_download_dir)
                    
                    local_download_checksum=local_download_config.get(upgrade_fortisoar_version,'checksum')

                    # Get the current checksum
                    checksum_cmd = "find '{0}' -type f -exec sha256sum {{}} + | sha256sum".format(local_download_dir)
                    actual_local_download_checksum=os.popen(checksum_cmd).read()

                    if actual_local_download_checksum.strip() != local_download_checksum.strip():
                        print("{}Warning{}:\nThe checksum values of the locally downloaded packages do not match.\nWould you like to perform upgrade using packages from the production repository [y/n] ?".format(TEXT_COLOR["YELLOW"],TEXT_COLOR["RESET"]))
                        user_input=self._yes_no_user_input()
                        if not user_input:
                            print("Exiting the upgrade")
                            print("Re-download the packages in local and try again")
                            sys.exit()
                        
                        # If yes then upgrade from the prod yum
                        flag_upgrade_from_local=False
                        # Set the flag_upgrade_from_local to False
                        local_download_config.set(str(upgrade_fortisoar_version),"flag_upgrade_from_local","False")
                        with open(self.LOCAL_PACKAGES_CONFIG_FILE, 'w') as configfile:
                            local_download_config.write(configfile)
                        # update result json
                        self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'flag_upgrade_from_local', flag_upgrade_from_local)
                        # Remove the local package dir
                        os.system(f"chattr -iR {local_download_dir}")
                        os.system(f"rm -rf {local_download_dir}") 

                    if flag_upgrade_from_local :
                        self._update_yum_repository_conf_files("create_bkp")
                
                self._upgrade_enterprise()

            elif flag_is_secure_message_exchange:
                self._upgrade_secure_message_exchange()
                
            cmd = f"rpm -q {s_pkg}"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            result = result['std_out']
            upgraded_cyops_common_build = result.split("-")[3].split(".")[0]
            self.store_result_to_json(self.store_result_path, 'upgrade', 'upgrade', 'build_number', upgraded_cyops_common_build)
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )

    # The upgrade_enterprise() function upgrade the FortiSOAR enterprise.
    def _upgrade_enterprise(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_pkg = step_result['s_pkg']
            custom_yum_url = step_result['custom_yum_url']
            l_pip_conf = step_result['l_pip_conf']
            
            upgrade_step_result = self.get_step_results('upgrade', 'upgrade')
            flag_upgrade_from_local = upgrade_step_result['flag_upgrade_from_local']
            
            self._cleanup_repos()

            os.system("yum clean all")

            # Disable the connectors YUM repository
            self._yum_repo("disable", "fsr-connectors*")
            
            cmd = f"rpm -q --queryformat '%{{version}}' {s_pkg}"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            s_installed_cyops_ver = result['std_out']
            
            # removing /usr/bin/pip3 since it was provided by python36
            # since python39 upgrades in 7.3.1, it creates a link from /usr/bin/pip3 -> /etc/alternatives/pip3
            # Doing this only in case of upgrade from 7.3.0
            # To be removed after 7.3.1 becomes baseline
            status = self._version_compare("7.3.1", s_installed_cyops_ver)
            if status == 1:
                os.system("rm -f /usr/bin/pip3")

            if flag_upgrade_from_local:
                # while upgrading from the local directory keep the cache rpms which will handle the pre-cleanup of the rpms
                cmd = f"yum update -y --setopt=keepcache=1 --exclude=fsr-elevate"
            else:
                cmd = "yum update -y --exclude=fsr-elevate"    
            
            print("RPM package upgrades are in progress and may take some time; please wait as the process completes.")
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            err = result['std_err']
            if return_code != 0:
                self.print_txt("{}ERROR{}: Failed to run yum update. Please refer '{}' file.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],LOG_FILE))
                if err:
                    self.logger.error(f"ERROR: {err}")
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                self._exit(1) 

            # The following condition checks for offline repo.
            # After yum update /opt/cyops-workflow/.env/pip.conf file if updated and it starts pointing to prod repo, so
            # if offline repo then update the pip.conf to offline repo
            if custom_yum_url:
                self._update_environment_file(custom_yum_url)
                for f_pip_conf in l_pip_conf:
                    result = self._update_pip_conf(f_pip_conf, custom_yum_url)
                    if not result:
                        self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                        self._exit(1)
                        
            self._cleanup_repos()
            
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )

    # The upgrade_secure_message_exchange() function upgrade the FortiSOAR secure-message-exchange.
    def _upgrade_secure_message_exchange(self):
        try:
            self._cleanup_repos()
            
            os.system("yum clean all")
            
            # Check if fsr-elevate is installed
            is_fsr_elevate_installed = False
            cmd = "rpm -qa fsr-elevate"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            output = result['std_out']
            if return_code == 0 and output:
                is_fsr_elevate_installed = True

            if is_fsr_elevate_installed:
                cmd = "yum update -y --exclude=fsr-elevate"
            else:
                cmd = "yum update -y"
            
            print("RPM package upgrades are in progress and may take some time; please wait as the process completes.")
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            err = result['std_err']
            if return_code != 0:
                self.print_txt("{}ERROR{}: Failed to run yum update. Please refer '{}' file.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],LOG_FILE))
                if err:
                    self.logger.error(f"ERROR: {err}")
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                self._exit(1)
            self._cleanup_repos()
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )

    # In 7.5.1, We are jump upgrading rabbitmq from 3.10 -> 3.11 -> 3.12 -> 3.13
    def _upgrade_rabbitmq_server(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_rabbitmq_intermediate_version_1 = step_result['s_rabbitmq_intermediate_version_1']
            s_rabbitmq_intermediate_version_2 = step_result['s_rabbitmq_intermediate_version_2']
            
            # While upgrading from rabbitmq 3.11 to 3.12, all the feature flags need to be enabled before upgrade.
            # While upgrading from 3.10 to 3.11, Some feature flags should be enabled. 
            # While upgrading from 3.11 to 3.12, All feature flags should be enabled.
            # Ref : https://www.rabbitmq.com/docs/upgrade#rabbitmq-version-upgradability 
            commands = ["rabbitmqctl enable_feature_flag all",
                        f"yum upgrade rabbitmq-server-{s_rabbitmq_intermediate_version_1} -y",
                        "rabbitmqctl enable_feature_flag all",
                        f"yum upgrade rabbitmq-server-{s_rabbitmq_intermediate_version_2} -y",
                        "yum upgrade rabbitmq-server -y",
                        "rabbitmqctl enable_feature_flag all"]
            return_code = self._execute_command_list(commands)
            if (return_code == -1):
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                self.print_txt("RabbitMQ server update failed")
                self._exit(1)
            self.print_txt("RabbitMQ server updated successfully")
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )

    # Function to execute list of commands
    def _execute_command_list(self, commands):
        for command in commands:
            try:
                self.print_txt(f"Executing : '{command}'")
                result = self.cmd_line_utilities.execute_cmd(command, True)
            except Exception as ex:
                err_msg = f"ERROR: Error in executing '{command}' : {ex}"
                self.logger.exception(err_msg)
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(
                    f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
                )
            return_code = result['return_code']
            err = result['std_err']
            
            if return_code != 0:
                self.print_txt("{}ERROR{}: An error occurred while executing '{}' command. Please refer '{}' file.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],command,LOG_FILE))
                if err:
                    err_msg = f"ERROR: Error in executing '{command}' : {err}"
                    self.logger.error(err_msg)
                return -1
        return 0

    def _update_yum_repository_conf_files(self, action):
        step_result = self.get_step_results('upgrade', 'upgrade')
        local_download_dir = step_result['local_download_dir']
        
        repo_conf_file_dir = "/etc/yum.repos.d"
        fsr_local_repo_file = "fsr-local.repo"
        conf_file_list = ["fsr-app.repo","fsr-os.repo","fsr-third-party.repo"]

        local_repo_config_lines_list = [
            "[fsr-local]\n",
            "name=FortiSOAR Local Repository\n",
            f"baseurl={local_download_dir}/{self.LOCAL_PACKAGES_DIR}\n",
            "gpgcheck=0\n",
            "enabled=1\n\n"

            "[fsr-connectors]\n",
            "name=FortiSOAR Connectors Local Repository\n"
            f"baseurl={local_download_dir}/{self.LOCAL_SYSTEM_CONNECTORS_DIR}\n",
            "gpgcheck=0\n",
            "enabled=1\n\n"
        ]

        if action=="create_bkp":
            # create the backup of the fsr yum repository conf files
            for conf_file in conf_file_list:
                os.system(f"mv  {repo_conf_file_dir}/{conf_file} {repo_conf_file_dir}/{conf_file}-bkp") 

            # create a local .repo file pointing to the local dir where the packages are downloaded
            with open(f"{repo_conf_file_dir}/{fsr_local_repo_file}", "w") as file:
                file.writelines(local_repo_config_lines_list)        

        elif action=="revert_bkp":
            # check if local .repo file exists 
            if os.path.exists(f"{repo_conf_file_dir}/{fsr_local_repo_file}"):
                for conf_file in conf_file_list:
                    os.system(f"mv  {repo_conf_file_dir}/{conf_file}-bkp {repo_conf_file_dir}/{conf_file}") 
                # remove the local .repo file
                os.remove(f"{repo_conf_file_dir}/{fsr_local_repo_file}")    

        else:
            print("Invalid argument!")    

    def _cleanup_repos(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            d_yum_repo = step_result['d_yum_repo']
            # to do : whether to remove the redhat repos or not
            redhat_repo_files = glob.glob(os.path.join(d_yum_repo,'redhat-*.repo'))
            rocky_repo_files = glob.glob(os.path.join(d_yum_repo,'rocky*.repo'))
            google_chrome_repo_files = glob.glob(os.path.join(d_yum_repo,'google-chrome.repo'))

            # Want the system point to FSR repos only
            self._remove_files_from_system(redhat_repo_files)
            self._remove_files_from_system(rocky_repo_files)
            self._remove_files_from_system(google_chrome_repo_files)
            
            os.system("yum clean all")
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at upgrade task. Refer logs at '{LOG_FILE}'"
            )
    
    def _remove_files_from_system(self,files: list) -> None:
        if files:
            for file in files:
                if isinstance(file,str) and os.path.exists(file):
                    os.remove(file)

    # Enable / Disable YUM Repository
    def _yum_repo(self, operation: str, repo: str):
        os.system(f"yum-config-manager --{operation} {repo} >/dev/null")

    # returns 0 if same,
    # returns 1 if version 1 is greater than version 2
    # returns 2 is version 1 is less than version 2
    def _version_compare(self, s_ver1: str, s_ver2: str):
        a_ver1 = s_ver1.split('.')
        a_ver2 = s_ver2.split('.')
        i_no_of_ele_ver1 = len(a_ver1)
        i_no_of_ele_ver2 = len(a_ver2)
        i_ver1 = int(s_ver1.replace('.', ''))
        i_ver2 = int(s_ver2.replace('.', ''))
        i_result = 0
        if i_no_of_ele_ver1 != i_no_of_ele_ver2:
            self.print_txt(
                "version_compare sub-routine needs identical number of elements")
            self._exit(1)

        if i_ver1 == i_ver2:
            # version 1, version 2 both are same
            i_result = 0
        elif i_ver1 > i_ver2:
            # version 1 is greater
            i_result = 1
        else:
            # version 2 is greater
            i_result = 2
        return i_result

    def _exit(self, exit_code: int):
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_bash_profile = step_result['f_bash_profile']
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))
        sys.exit(exit_code)
    
    # The update_environment_file function updates entry in the file /etc/environment with custom_yum_url variable value.
    # Also, this function calls in cleanup function because cyops rpm package added the entry of \
    # product_yum_server=repo.fortisoar.fortinet.com. So to change the value of product_yum_server variable \
    # update_environment_file function calls in cleanup function.
    def _update_environment_file(self, custom_yum_url):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            f_etc_environment = step_result['f_etc_environment']
            cmd = "grep -qw {} {}".format(custom_yum_url, f_etc_environment)
            ret_code = os.system(cmd)
            if ret_code != 0:
                cmd = "sed -i \"/product_yum_server/c\product_yum_server={}\" {}".format(
                    custom_yum_url, f_etc_environment)
                result = self.cmd_line_utilities.execute_cmd(cmd)
                i_flag_status = result['return_code']
                if i_flag_status != 0:
                    self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
                    msg = "Failed to update the \"{f_etc_environment}\" file.\nCheck if the \"{f_etc_environment}\" file has write permission.".format(
                        f_etc_environment=f_etc_environment)
                    print(msg)
                    err_msg = "Failed to update the \"{f_etc_environment}\" file. Check if the \"{f_etc_environment}\" file has write permission.".format(
                        f_etc_environment=f_etc_environment)
                    self.logger.error(f"ERROR: {err_msg}")
                    return
                msg = "Successfully updated file '{}' with '{}'".format(f_etc_environment, custom_yum_url)
                self.logger.info(msg)
                self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )
    
    def _update_pip_conf(self, f_pip_conf, custom_yum_url):
        try:
            # The following tickets have details to add update_pip_conf function.
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0793364
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0771138
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0771134
            s_trusted_host = "trusted-host"

            if not os.path.exists(f_pip_conf):
                err_msg = "The '{}' does not found.".format(f_pip_conf)
                self.print_txt(err_msg)
                self.logger.error(f"ERROR: {err_msg}")
                return False
            
            cmd = "grep -wq '{}' {}".format(custom_yum_url, f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            if return_code == 0:
                msg = "{} already present in {}.".format(
                    custom_yum_url, f_pip_conf)
                self.print_txt(msg)
                return True
            
            cmd = "grep -w \"{}\" {}".format(s_trusted_host, f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd)
            is_trusted_host_exist = result['return_code']
            
            local_custom_repo_name = custom_yum_url
            local_custom_yum_url = local_custom_repo_name
            if not local_custom_yum_url.startswith("https://"):
                local_custom_yum_url = "https://{}".format(local_custom_yum_url)
            
            # Making pip.conf file mutable for updating the file
            os.system(f"chattr -i {f_pip_conf}")
            # Install the connector if install from offline repo.
            # If trusted-host not exit in pip.conf files then connector installation fails.
            if is_trusted_host_exist != 0:
                with open(f_pip_conf, 'a') as conf_file:
                    conf_file.write('{}={}'.format(s_trusted_host, local_custom_repo_name))
            
            url_pattern = "http[s]:\/\/(?:[a-zA-z0-9]+\.)+com"
            cmd = "grep -w \"extra-index-url\" {}".format(f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            return_code = result['return_code']
            s_repo_name = ""
            if return_code == 0:
                std_out = result['std_out']
                s_repo_name = re.findall(url_pattern, std_out)
                s_repo_name = s_repo_name[0] if s_repo_name else ""

            if s_repo_name:
                # custom yum url not found in pip conf file, updating pip conf file
                cmd = "sed -i 's|{}|{}|g' {}".format(s_repo_name, local_custom_yum_url, f_pip_conf)
                result = self.cmd_line_utilities.execute_cmd(cmd)
                return_code = result['return_code']

                if return_code != 0:
                    err_msg = "Failed to update the file {}.".format(f_pip_conf)
                    self.print_txt(err_msg)
                    self.logger.error(f"ERROR: {err_msg}")
                    # Making pip.conf file immutable if failed to update the file
                    os.system(f"chattr +i {f_pip_conf}")
                    return False
            # Making pip.conf file immutable after updating the file
            os.system(f"chattr +i {f_pip_conf}")
            return True
        except Exception as ex:
            # Making pip.conf file immutable if exception occurs
            os.system(f"chattr +i {f_pip_conf}")
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )