""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
Copyright end """
import os
import sys
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE
import random
import string
import yaml

try:
    # This import will fail in case of external SME
    sys.path.append("/opt/cyops-auth")
    from utilities.ha.node import is_primary, is_secondary
    from utilities.ha.consts import BKP_CYOPS_CONN_FILE, CONF_FILE, DEFAULT_CYOPS_CONN_FILE, SUSPEND_CLUSTER_BACKUP_DIR_TOP
    from utilities.ha.conf import read_config_file
except ImportError:
    pass


TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED": "STARTED", "COMPLETED": "COMPLETED"}
TEXT_COLOR = {'GREEN': '\033[92m', 'RED': '\033[91m', 'YELLOW': '\033[93m', 'RESET': '\033[0m'}
TEXT_DECORATION = {'BLINK': '\033[5m', 'BOLD': '\033[1m', 'RESET': '\033[0m'}


class UpdateSecretKeysUpgradeTask(Tasks):
    TASK_STATUS_MSG = "Update postman key"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return "Updates postman key"

    def is_supported(self) -> bool:
        try:
            current_version = int(self.current_version.replace('.', ''))
            supported_version = 763
            target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
            return target_upgrade_version >= current_version and self.is_enterprise() and current_version < supported_version
        except ValueError as e:
            self.logger.error(f"Version parsing error: {e}")
            return False

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["STARTED"])
        self.update_postman_key()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        msg = "    " + msg
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]

        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status_text = f"{status:^{width}}"
        colored_status = f"{color}{status_text}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg: str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status, ":", msg)
        border_length = len(status_msg)
        border = '=' * border_length
        new_line_char = "\n" if status == TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        try:
            with open(LOG_FILE, 'a') as log_file:
                log_file.write(final_msg)
        except Exception as e:
            self.logger.error(f"Failed to write log file: {e}")

    def update_postman_key(self):
        try:
            self.update_secret_in_config()
            if self.is_secondary_node():
                self.update_secret_in_config(DEFAULT_CYOPS_CONN_FILE)

            # If cluster is suspended, update in backup config file
            bkp_config_file = SUSPEND_CLUSTER_BACKUP_DIR_TOP + "/" + "db_config.yml"
            self.update_secret_in_config(bkp_config_file)

            self.cmd_line_utilities.execute_cmd("cmd", True)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['DONE'])
        except Exception as ex:
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            err_msg = f"Failed to update secret keys: {ex}"
            self.logger.exception(err_msg)
            print(f"Exception occurred in post-upgrade task. Check logs at '{LOG_FILE}'")


    def update_secret_in_config(self, conf_file=CONF_FILE):
        try:
            if not os.path.exists(conf_file):
                self.logger.warning(f"File not found: {conf_file}")
                return

            config = read_config_file(conf_file)
            if config.get('verification_key') and config.get('verification_key').get('router'):
                return

            postman_key = self.get_random_string()
            config['verification_key'] = {
                'router': postman_key
            }
            update_current_config(config, conf_file)
            self.logger.warning(f"Completed : {conf_file}")
        except Exception as error:
            self.logger.exception(error)
            raise


    def get_random_string(self, length=30):
        characters = string.ascii_letters + string.digits  # [A-Za-z0-9]
        return ''.join(random.choice(characters) for _ in range(length))


    def is_enterprise(self):
        step_result = self.get_step_results("pre-upgrade", "initialize")
        return step_result["flag_is_enterprise"]

    def is_secondary_node(self):
        try:
            if self.is_enterprise():
                return is_secondary()
            return False
        except Exception as error:
            self.logger.exception(error)
            return False

def update_current_config(config, file_path=CONF_FILE):
    with open(file_path, 'w') as config_file:
        yaml.dump(config, config_file, default_flow_style=False)
