""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
Copyright end """

from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED": "STARTED", "COMPLETED": "COMPLETED"}
TEXT_COLOR = {'GREEN': '\033[92m', 'RED': '\033[91m', 'YELLOW': '\033[93m', 'RESET': '\033[0m'}
TEXT_DECORATION = {'BLINK': '\033[5m', 'BOLD': '\033[1m', 'RESET': '\033[0m'}


class InstallWorkflowDependencyTask(Tasks):
    TASK_STATUS_MSG = "Install workflow dependencies"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return "Install workflow dependency"

    def is_supported(self) -> bool:
        try:
            current_version = int(self.current_version.replace('.', ''))
            target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
            return target_upgrade_version >= current_version and self.is_enterprise()
        except ValueError as e:
            self.logger.error(f"Version parsing error: {e}")
            return False

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["STARTED"])
        cmd = 'sudo -u nginx /opt/cyops-workflow/.env/bin/pip3 install html2text'
        result = self.cmd_line_utilities.execute_cmd(cmd, trap_output=True)
        if result.get('std_out'):
            self.logger.info(result.get('std_out'))
        if result.get('return_code') != 0:
            self.logger.error("{0} failed with error: {1}".format(self.TASK_STATUS_MSG, result.get('std_err')))
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['FAILED'])
        else:
            self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        msg = "    " + msg
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]

        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status_text = f"{status:^{width}}"
        colored_status = f"{color}{status_text}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg: str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status, ":", msg)
        border_length = len(status_msg)
        border = '=' * border_length
        new_line_char = "\n" if status == TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        try:
            with open(LOG_FILE, 'a') as log_file:
                log_file.write(final_msg)
        except Exception as e:
            self.logger.error(f"Failed to write log file: {e}")

    def is_enterprise(self):
        step_result = self.get_step_results("pre-upgrade", "initialize")
        return step_result["flag_is_enterprise"]
