""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """
import subprocess
import time

from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from helpers.logger import Logger
from constants import REPORT_FILE, STEP_RESULT_FILE
import os
import re

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class ResetEmbeddedSMEPassword(Tasks):
    TASK_STATUS_MSG = "Updating RabbitMQ password for user {}"


    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'post_upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version and self.__is_cloud_instance() and self.__is_sme_embedded()

    def execute(self) -> None:
        admin_user = "admin"
        admin_password_old = "changeme"
        admin_password = self.__get_device_uuid()
        # Check if admin user exists
        check_user_cmd = "rabbitmqctl --no-table-headers --quiet list_users | awk {'print $1'} | grep -w admin"
        process = subprocess.run(args=check_user_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        ret = process.returncode
        std_out = process.stdout.strip().decode("utf-8")
        std_err = process.stderr.strip().decode("utf-8")
        if ret != 0:
            self.logger.warning(f"RabbitMQ user {admin_user} does not exist")
            return

        if not self.__rabbitmq_change_password(admin_user,admin_password):
            self.logger.warning(f"Could not change password for {admin_user}")
        else:
            self.logger.info(f"Performed change password for {admin_user}")
            # Now that password is changed, update the routers
            payload = {"password": admin_password}
            retry_count = 5
            while retry_count > 0:
                response = self.task_utilities._make_cyops_request("https://localhost/api/3/routers/52c5cee8-5c28-4ed2-a886-ec8bf4dc5993",method='put',payload=payload)
                if response.status_code == 200:
                    # Update the rabbitmq_users.conf file with updated password
                    rmq_user_file = '/opt/cyops/configs/rabbitmq/rabbitmq_users.conf'
                    encrypted_password = subprocess.check_output(
                        ['python3', '/opt/cyops/scripts/manage_passwords.py', '--encrypt',
                         admin_password]).decode().strip()

                    with open(rmq_user_file, 'r') as file:
                        file_contents = file.read()

                    # Replace the line containing 'cyops.rabbitmq.mgmtpassword'
                    new_contents = re.sub(r'cyops\.rabbitmq\.mgmtpassword=.*',
                                          f'cyops.rabbitmq.mgmtpassword="{encrypted_password}"', file_contents)

                    # Write the modified contents back to the file
                    with open(rmq_user_file, 'w') as file:
                        file.write(new_contents)
                    self.logger.info(f"Secure Message Exchange server has been changed to reflect the new admin password.")
                    self._print_status_msg(self.TASK_STATUS_MSG.format(admin_user), TASK_STATUS["DONE"])
                    return
                self.logger.warning(f"Failed to update routers with status {response.status_code}.Retrying in 2 seconds")
                retry_count -= 1
                time.sleep(2)

            # Revert the password back to changeme if we are unable to update the routers
            self.__rabbitmq_change_password(admin_user, admin_password_old)
            self._print_status_msg(self.TASK_STATUS_MSG.format(admin_user), TASK_STATUS["FAILED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg: str, status: str) -> None:
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def __is_cloud_instance(self):
        is_cloud_instance = False
        if os.path.exists('/etc/cyops-release'):
            with open('/etc/cyops-release') as product_type_file:
                if 'forticloud' in product_type_file.read():
                    is_cloud_instance = True
        return is_cloud_instance

    def __is_sme_embedded(self) -> bool:
        if os.path.exists("/etc/cyops-release"):
            with open("/etc/cyops-release") as product_type_file:
                if "secure-message-exchange" in product_type_file.read():
                    return True
        return False

    def __get_device_uuid(self):
        get_device_uuid_cmd = "csadm license --get-device-uuid"
        device_uuid = self.cmd_line_utilities.execute_cmd(get_device_uuid_cmd,True)
        return device_uuid['std_out'].strip()

    def __rabbitmq_change_password(self,user,password):
        change_password_cmd = "rabbitmqctl change_password {0} '{1}'".format(user, password)
        process = subprocess.run(args=change_password_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        ret = process.returncode
        std_out = process.stdout.strip().decode("utf-8")
        std_err = process.stderr.strip().decode("utf-8")

        if ret != 0:
            self.logger.error(f"Failed to update RabbitMQ password for user {user} {std_err}")
            return False
        return True
