""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys

import psutil
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import (
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE
)


TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}
RECOMMENDED_VCPU_COUNT = 12
RECOMMENDED_RAM_SIZE = 48

class CheckSystemRequirement(Tasks):
    TASK_STATUS_MSG = "Validate required system resources"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return "pre-upgrade"

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace(".", ""))
        target_upgrade_version = int(self.target_upgrade_version.replace(".", ""))
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        return target_upgrade_version > current_version and flag_is_enterprise

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        result = self.check_system_requirements()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])
        return result

    def _yes_no_user_input(self):
        yes_list = ["y","Y","Yes","yes","YES"]
        no_list = ["n","N","No","no","NO"]
        i_count=0
        while i_count <=2:
            user_input = input()
            if user_input in yes_list:
                return True
            elif user_input in no_list:
                return False
            else:
                print("You have provided an invalid input. Enter Yes or No")
                i_count+=1
        print("Max retries reached, exiting the upgrade.")
        return False

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def check_system_requirements(self):
        cpu_check_passed, ram_check_passed = True, True
        errors = []

        current_vcpu = CheckSystemRequirement._get_vcpu_count()
        current_ram = self._get_ram_size()

        if current_vcpu < RECOMMENDED_VCPU_COUNT:
            cpu_check_passed = False
            errors.append(
                f"- vCPU count: {current_vcpu} (minimum required: {RECOMMENDED_VCPU_COUNT})"
            )

        if current_ram < RECOMMENDED_RAM_SIZE:
            ram_check_passed = False
            errors.append(
                f"- RAM size: {current_ram} GB (minimum required: {RECOMMENDED_RAM_SIZE} GB)"
            )

        result = cpu_check_passed and ram_check_passed

        if result:
            success_msg = "Minimum system requirements have been met."
            self.store_result_to_json(
                self.report_path, None, None, self.TASK_STATUS_MSG,
                {"result": True, "message": success_msg}
            )
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['DONE'])
            return True

        # If requirements not met
        error_header = "Minimum system requirements not met. Please review:"
        report_msg = f"{error_header}\n" + "\n".join(errors)

        self.logger.error(report_msg)
        self.store_result_to_json(
            self.report_path, None, None, self.TASK_STATUS_MSG,
            {"result": False, "message": report_msg}
        )
        self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['FAILED'])
        print(report_msg)

        # Ask user if they still want to proceed
        print(f"\n{TEXT_COLOR['YELLOW']}Warning{TEXT_COLOR['RESET']}: "
              f"Do you still want to proceed with upgrade? [y/n]")

        validate_result = self._yes_no_user_input()

        if not validate_result:
            sys.exit()
        return True

    @staticmethod
    def _get_vcpu_count():
        """
        Returns the number of virtual (logical) CPUs.
        """
        return psutil.cpu_count(logical=True)

    def _get_ram_size(self):
        try:
            cmd = "lsmem --summary=only -b | grep 'Total online memory:' | awk -F': ' '{print $2}' | tr -d ' '"
            mem_bytes = int(subprocess.check_output(cmd, shell=True))
            return mem_bytes / (1024 ** 3)
        except Exception as err:
            self.logger.error("Error occurred while getting RAM size. Error: {0}".format(err))
            return 0