""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import glob
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class UpdateSystemConnectors(Tasks):
    TASK_STATUS_MSG = "Update system connectors"
    LOCAL_SYSTEM_CONNECTORS_DIR = "fsr-connectors"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            upgrade_step_result = self.get_step_results('upgrade', 'upgrade')
            flag_upgrade_from_local = upgrade_step_result['flag_upgrade_from_local']
            local_download_dir=upgrade_step_result['local_download_dir']
            
            # Check if flag_upgrade_from_local is true and  connectors are locally downloaded  
            if flag_upgrade_from_local and os.path.exists(f"{local_download_dir}/{self.LOCAL_SYSTEM_CONNECTORS_DIR}"):
                self._update_system_connectors()
            
            # Check if flag_upgrade_from_local is true and connectors are not locally downloaded
            # Means system connectors are upto date
            elif flag_upgrade_from_local and not os.path.exists(f"{local_download_dir}/{self.LOCAL_SYSTEM_CONNECTORS_DIR}"):  
                print("System connectors are up-to-date")
            else:
                # else upgrade pointing to yum repo
                self._update_system_connectors()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _update_system_connectors(self):
        try:
            # Enable the connectors YUM repository
            self._yum_repo("enable", "fsr-connectors*")
            
            self._cleanup_repos()

            connectors_list = ["cyops-connector-cyops_utilities", "cyops-connector-imap", "cyops-connector-smtp", "cyops-connector-cyops-schedule-report", "cyops-connector-code-snippet",
                               "cyops-connector-cyops-system-monitoring", "cyops-connector-fortisoar-ml-engine", "cyops-connector-fsr-agent-communication-bridge", "cyops-connector-phishing-classifier"]
            
            for connector in connectors_list:
                cmd = f"yum update -y {connector}"
                result = self.cmd_line_utilities.execute_cmd(cmd)
                return_code = result['return_code']
                err = result['std_err']
                if return_code != 0:
                    if err:
                        self.logger.error(f"ERROR: {err}")
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    print(f"Please refer {LOG_FILE} file")
                    self._exit(1)
            
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update system connectors task. Refer logs at '{LOG_FILE}'"
            )

    def _cleanup_repos(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            d_yum_repo = step_result['d_yum_repo']
            redhat_repo_files = glob.glob(os.path.join(d_yum_repo,'redhat-*.repo'))
            rocky_repo_files = glob.glob(os.path.join(d_yum_repo,'Rocky-*.repo'))
            google_chrome_repo_files = glob.glob(os.path.join(d_yum_repo,'google-chrome.repo'))

            # Want the system point to FSR repos only
            self._remove_files_from_system(redhat_repo_files)
            self._remove_files_from_system(rocky_repo_files)
            self._remove_files_from_system(google_chrome_repo_files)
            
            os.system("yum clean all")
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update system connectors task. Refer logs at '{LOG_FILE}'"
            )

    def _remove_files_from_system(self,files: list) -> None:
        if files:
            for file in files:
                if isinstance(file,str) and os.path.exists(file):
                    os.remove(file)
    
    # Enable / Disable YUM Repository
    def _yum_repo(self, operation: str, repo: str):
        os.system(f"yum-config-manager --{operation} {repo} >/dev/null")
        
    def _exit(self, exit_code: int):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        os.system("sed -i '/config-vm.sh$/d' $f_bash_profile")
        os.system("sed -i '/check_eula.sh$/d' $f_bash_profile")
        sys.exit(exit_code)