""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class PackageManagement(Tasks):
    TASK_STATUS_MSG = "Manage nodejs packages"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'upgrade'

    def get_description(self) -> str:
        return "This task will manage packages like nodejs etc."

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._manage_packages()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _manage_packages(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_nodejs_ver = step_result['s_nodejs_ver']
            commands = ["yum module reset nodejs -y", "yum module list nodejs -y",
                            f"yum module enable nodejs:{s_nodejs_ver} -y", "yum update nodejs -y", "yum -y install fio ioping"]
            for command in commands:
                try:
                    self.print_txt(f"Executing : '{command}'")
                    result = self.cmd_line_utilities.execute_cmd(command, True)
                    return_code = result['return_code']
                    err = result['std_err']
                    if err:
                        err_msg = f"Error in executing '{command}' : {err}"
                        self.logger.error(f"ERROR: {err_msg}")
                        self.print_txt("{}ERROR{}: ".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],err_msg))
                except Exception as ex:
                    err_msg = f"ERROR: An error occurred while executing '{command}' command : {ex} "
                    self.logger.exception(err_msg)
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    print(f"Exception occurred at package management task. Refer logs at '{LOG_FILE}'")

                if return_code != 0:
                    if err:
                        self.logger.error(f"ERROR: {err}")
                    print("{}ERROR{}: An error occurred while executing '{}' command. Please refer {} file.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],command,LOG_FILE))
                    self._exit(1)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at package management task. Refer logs at '{LOG_FILE}'"
            )

    def _exit(self, exit_code: int):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        os.system("sed -i '/config-vm.sh$/d' $f_bash_profile")
        os.system("sed -i '/check_eula.sh$/d' $f_bash_profile")
        sys.exit(exit_code)