""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from helpers.logger import Logger
from constants import REPORT_FILE, STEP_RESULT_FILE
import os

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class RemoveLoopbackUser(Tasks):
    TASK_STATUS_MSG = "Remove users from loopback interface for RabbitMQ."


    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'post_upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version and self.__is_cloud_instance()

    def execute(self) -> None:
        users_to_remove = ['fsr-cluster', 'admin']
        if not self.__remove_mq_loopback_users(users_to_remove):
            err_msg = f"Failed to remove users {users_to_remove} from loopback"
            self.logger.error(f"ERROR: {err_msg}")
            self._print_status_msg(self.TASK_STATUS_MSG.format(users_to_remove), TASK_STATUS["FAILED"])
            print(err_msg)
        else:
            self._print_status_msg(self.TASK_STATUS_MSG.format(users_to_remove), TASK_STATUS["DONE"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg: str, status: str) -> None:
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def __remove_mq_loopback_users(self,users_to_remove: list):
        mq_conf = '/opt/cyops-rabbitmq/configs/ssl/rabbitmq-embedded-sme.conf'
        removed_users = set()
        if not os.path.isfile(mq_conf):
            self.logger.error("File [" + mq_conf + "] does not exist.")
            return True
        try:
            with open(mq_conf, 'r') as file:
                lines = file.readlines()
            with open(mq_conf, 'w') as file:
                for line in lines:
                    found_in_line = False
                    for user in users_to_remove:
                        if f"loopback_users.{user}" in line.strip():
                            found_in_line = True
                            removed_users.add(user)
                    if not found_in_line:
                        file.write(line)

        except Exception as err:
            self.logger.error(str(err))
            return False
        self.logger.info("Removed users {} from {}".format(removed_users,mq_conf))
        return True

    def __is_cloud_instance(self):
        is_cloud_instance = False
        if os.path.exists('/etc/cyops-release'):
            with open('/etc/cyops-release') as product_type_file:
                if 'forticloud' in product_type_file.read():
                    is_cloud_instance = True
        return is_cloud_instance