""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class ReservePgsqlDiskSpace(Tasks):
    TASK_STATUS_MSG = "Reserve pgsql disk space"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._reserve_pgsql_disk_space()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True
    
    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    # The reserve_pgsql_disk_space() function reserve the 1 GB space, in case disk full issue the user,
    # will delete the file /var/lib/pgsql/resv_space and utilize the space
    # when require and create the file again using the following command once issue resolve.
    def _reserve_pgsql_disk_space(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            d_pgsql_path = step_result['d_pgsql_path']
            f_pgsql_resv_space_file = step_result['f_pgsql_resv_space_file']
            var_post_upgrade_msg = step_result['var_post_upgrade_msg']

            resv_pg_file = f"{d_pgsql_path}/{f_pgsql_resv_space_file}"
            if resv_pg_file and os.path.exists(resv_pg_file):
                # Space is already reserved
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
                return

            # The following command is dd if=/dev/zero of=/var/lib/pgsql/resv_space count=1024 bs=1048576
            cmd = f"dd if=/dev/zero of={resv_pg_file} count=1024 bs=1048576"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            return_code = result['return_code']
            if return_code != 0:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                self.print_txt(
                    f"Failed to create the {resv_pg_file} file. Make sure {d_pgsql_path} has sufficient space.\nPlease re-run the following command manually.")
                self.print_txt(
                    f"dd if=/dev/zero of={resv_pg_file} count=1024 bs=1048576")
                var_post_upgrade_msg = var_post_upgrade_msg.append(
                    f"Failed to create the {resv_pg_file} file. Make sure {d_pgsql_path} has sufficient space. Please re-run the command manually : dd if=/dev/zero of={resv_pg_file} count=1024 bs=1048576")
                step_result['var_post_upgrade_msg'] = var_post_upgrade_msg
                self.store_result_to_json(
                    self.store_result_path, 'pre-upgrade', 'initialize', None, step_result)
            self.logger.info(f"Reserved 1 GB space for postgresql in {d_pgsql_path}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at reserve pgsql disk space task. Refer logs at '{LOG_FILE}'"
            )