""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import (
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE
)

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class VerifyFilesystemMount(Tasks):
    TASK_STATUS_MSG = "Verify NFS mount"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return "pre-upgrade"

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace(".", ""))
        target_upgrade_version = int(self.target_upgrade_version.replace(".", ""))
        return target_upgrade_version > current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        self.check_nfs_mount()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True
    
    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
    
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
        
    def _yes_no_user_input(self):
        yes_list = ["y","Y","Yes","yes","YES"]
        no_list = ["n","N","No","no","NO"]
        i_count=0
        while i_count <=2:
            user_input = input()
            if user_input in yes_list:
                return True
            elif user_input in no_list:
                return False
            else:
                print("You have provided an invalid input. Enter Yes or No")   
                i_count+=1 
        if  i_count > 2:
            print("Max retries reached, exiting the upgrade.")    

    def check_nfs_mount(self):
        # check if nsf is mounted
        cmd = "grep nfs /proc/mounts"
        result = self.cmd_line_utilities.execute_cmd(cmd, True)
        if result['return_code'] == 0:
            nfs_detect_msg = "NFS mount point detected."
            self.logger.info(nfs_detect_msg)
            # check if nsf is stale
            # if nsf is stale then df command gets stuck 
            cmd = "timeout 30 df"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            
            if result['return_code'] != 0:
                self.logger.info("The NFS-mounted filesystem is detected to be stale")
                print("{}Warning{}:\nUpgrading FortiSOAR on an NFS-mounted partition might result in upgrade failure and hence is not recommended.\nDo want to still proceed with upgrade [y/n] ?".format(TEXT_COLOR["YELLOW"],TEXT_COLOR["RESET"]))
                user_input=self._yes_no_user_input()
                if not user_input:
                    print("Exiting the upgrade")
                    self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["DONE"])
                    sys.exit()              
        self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["DONE"])
                    